<?php

namespace App\Helpers;

use App\Enums\MetaFileExtensionEnum;
use App\Models\Message;
use App\Models\MessageCampaignUser;
use App\Models\MessageTemplate;
use App\Models\MetaLog;
use App\Models\Setting;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class MetaWebhook
{
    /**
     * @var string
     */
    protected $wpBusinessAccountId;

    /**
     * @var string
     */
    protected $phoneNumberId;

    /**
     * MetaWebhook constructor.
     * @throws \Exception
     */
    public function __construct()
    {
        $settings = Setting::first();
        $this->wpBusinessAccountId = $settings->whatsapp_meta_business_account_id;
        $this->phoneNumberId = $settings->whatsapp_meta_phone_number_id;
        if (empty($this->wpBusinessAccountId) || empty($this->phoneNumberId)) {
            throw new \Exception("whats app configuration missing");
        }
    }

    public function log()
    {
        return Log::class;
        
        // "error": "The stream or file "/var/www/html/vdoconfrence/storage/logs/meta-2024-10-30.log 
        //could not be opened in append mode: Failed to open stream: Permission denied\nThe exception occurred while attempting to log
        // return Log::channel("meta");
    }

    public function parse($data)
    {
        $wpBusinessAccountId = $this->wpBusinessAccountId;

        $entries = $data["entry"] ?? [];
        foreach ($entries as $entry) {
            $metaId = $entry["id"] ?? null;
            if ($metaId !== $wpBusinessAccountId) {
                $this->log()::error("MetaHook: Invalid business Account Id: {$metaId}");
            } else {
                $this->parseEntry($entry);
            }
        }
    }

    protected function parseEntry($entry)
    {
        $changes = $entry["changes"] ?? [];
        foreach ($changes as $change) {
            $this->parseChange($change);
        }
    }

    protected function parseChange($change)
    {
        MetaLog::create([
            "name" => "webhook_" . $change["field"] ?? "null",
            "request" => $change
        ]);

        if ($change["field"] === "message_template_status_update") {
            $this->handleMessageTemplateStatus($change);
        } else if ($change["field"] === "messages") {

            $this->handleMessageStatus($change);
        } else {
            $this->log()::warning("MISSING FIELD FOUND: " . json_encode($change));
        }
    }

    protected function handleMessageTemplateStatus($change)
    {
        $templateId = $change["value"]["message_template_id"] ?? null;
        if ($templateId && $template = MessageTemplate::where("template_id", $templateId)->first()) {
            $template->status = $change["value"]["event"] ?? "PENDING";
            $template->save();
        } else {
            $this->log()::warning("MISSING TEMPLATE FOUND: " . json_encode($change));
        }
    }

    protected function handleMessageStatus($change)
    {
        $statues = $change["value"]["statuses"] ?? [];
        foreach ($statues as $changeStatus) {
            $this->updateMessageStatus($changeStatus["id"], $changeStatus["status"]);
        }

        $messages = $change["value"]["messages"] ?? [];
        foreach ($messages as $message) {
            $this->addMessage($message);
        }
    }

    protected function updateMessageStatus($messageId, $status)
    {
        $message = Message::where('message_id', $messageId)->first();
        if ($message && method_exists($message, $status)) {
            $message->$status();
        }

        $campaignUser = MessageCampaignUser::where("meta_message_id", $messageId)->first();
        if ($campaignUser && method_exists($campaignUser, $status)) {
            $campaignUser->$status();
            if ($campaignUser->campaign) {
                $campaignUser->campaign->handleStatus($campaignUser->user_id, $status);
            }
        }
        $this->log()::error("MetaHook: Message Not Found: {$messageId}");
        return;
    }

    protected function addMessage($message)
    {
        $from = $message['from'] ?? null;
        Log::info("addMessage function-------" . $from);
        $user = User::where(DB::raw('CONCAT_WS("", country_code, contact)'), '+' . $from)->first();

        if (!$user) {
            $this->log()::error("MetaHook {$from}: USER NOT AVAILABLE");
            return;
        }

        $type = $message['type'] ?? null;
        $body = null;
        if (!$type) {
            $this->log()::error("MetaHook: EMPTY TYPE RECEIVED");
            return;
        }
        if ($type == "text") {
            $body = $message['text']['body'] ?? null;
        } else if ($type == "image" || $type == "video" || $type == "audio" || $type == "document" || $type == "sticker") {
            $mediaId = $message[$type]['id'] ?? null;
            if (!$mediaId) {
                $this->log()::error("MetaHook: Media ID NOT FOUND " . $message['id']);
                return;
            }
            $mediaResponse = MetaClient::api()->getMediaId($mediaId);
            $mediaUrl = $mediaResponse['url'] ?? null;
            if (!$mediaUrl) {
                $this->log()::error("MetaHook: MEDIA URL NOT FOUND " . $message['id']);
                return;
            }
            $mediaPath = MetaClient::api()->getMedia($mediaUrl);
            $base64Media = base64_encode($mediaPath->body());
            $mimeType = $message[$type]['mime_type'];
            $mimeTypeArray = explode("/", $mimeType);
            if (!isset($mimeTypeArray[1])) {
                $this->log()::error("MetaHook: MIME TYPE NOT FOUND " . $message['id']);
                return;
            }
            $mimeExtention = strtok(trim($mimeTypeArray[1]), ';');
            $mediaName = "whatsapp_media/" . $mediaId . "." . $mimeExtention;
            if (in_array($mimeType, MetaFileExtensionEnum::IMAGE)) {
                $body = '<img style="height: 150px;width: 150px;" src="' . asset("storage/" . $mediaName) . '">';
            } else if (in_array($mimeType, MetaFileExtensionEnum::VIDEO)) {
                $body = '<video width="150" height="150" controls><source src="' . asset("storage/" . $mediaName) . '" type="video/mp4"></video>';
            } else if (in_array($mimeType, MetaFileExtensionEnum::AUDIO)) {
                $body = '<audio controls style="width: 200px;"><source src="' . asset("storage/" . $mediaName) . '" type="audio/wav"></audio>';
            } else {
                $body = "<a href=" . asset("storage/" . $mediaName) . " download><i class='fa fa-file' aria-hidden='true' style='font-size:21px;margin-right: 10px;'></i>Document</a>";
            }
            Storage::put($mediaName, base64_decode($base64Media));
        }
        if (!$body) {
            $this->log()::error("MetaHook: EMPTY MESSAGE RECEIVED");
            return;
        }
        $contextId = $message["context"]["id"] ?? null;
        $reply = null;
        if ($contextId) {
            $reply = MessageCampaignUser::where("meta_message_id", $contextId)->first();
            if ($reply && !$reply->replied_at) {
                $reply->replied_at = Carbon::now()->toDateTimeString();
                $reply->save();
            }
            $reply = $reply ?? Message::where("message_id", $contextId)->first();
        }

        $messageTemplateId = null;

        Message::create([
            "sender_id" => $user->id,
            "message_template_id" => $messageTemplateId,
            "message_id" => $message['id'],
            "message" => $body,
            "replied_message_id" => $reply ? $reply->id : null
        ]);

        $this->log()::info("MetaHook: MESSAGE ID " . $message['id']);
    }
}
