<?php

namespace App\Helpers;

use App\Models\User;
use Illuminate\Support\Facades\Log;
use Razorpay\Api\Api;

class RazorPay implements PaymentInterface
{
    protected $baseUrl;

    protected $appKey;

    protected $appSecret;

    public function __construct($appKey, $appSecret)
    {
        $this->baseUrl = env("RAZORPAY_URL", "https://api.razorpay.com");

        $this->appKey = $appKey;

        $this->appSecret = $appSecret;
    }

    public function createOrder($eventName, $price, User $user, $data)
    {
        $api = new Api($this->appKey, $this->appSecret);

        $order = $api->order->create([
            'amount' => $price * 100,
            'currency' => "INR",
            'notes' => [
                "eventName" => $eventName,
                "price" => $price,
                "contact" => $user->contact,
                "name" => $user->name,
                "email" => $user->email,
            ],
        ]);

        return $order;
    }

    public function createExternalOrder($price, $product, $callBackUrl = null)
    {
        $api = new Api($this->appKey, $this->appSecret);
        $order = $api->order->create([
            'amount' => $price * 100,
            'currency' => "INR",
            'notes' => $product,
        ]);

        return $order;
    }

    /**
     * @param $paymentInfo
     * @return array
     * @throws \Exception
     */
    public function getPaymentDetails($paymentInfo) : array
    {
        $api = new Api($this->appKey, $this->appSecret);
        $payment =  $api->payment->fetch($paymentInfo["razorpay_payment_id"]);
        Log::info("Event Thanks: Payment Fetched - " . $paymentInfo["razorpay_payment_id"]);
        return [
            "signature" => $paymentInfo["razorpay_signature"],
            "payment_id" => $paymentInfo["razorpay_payment_id"],
            "amount" => $payment['amount'] / 100,
            "status" => "paid",
        ];
    }

    public function getOrderDetails($orderId)
    {
        $api = new Api($this->appKey, $this->appSecret);

        $order = $api->order->fetch($orderId);

        $response = $order->toArray();

        $payments = $order->payments()->toArray();
        $response["payments"] = $payments["items"];

        return $response;
    }

    public function parseOrderInfo($order)
    {
        $payments = $order["payments"] ?? [];
        return [
            "id" => $order["id"],
            "event_code" => $order["notes"]["eventName"],
            "amount" => $order["amount_paid"]/100,
            "status" => $order["status"] === "paid" ? "paid" : "pending",
            "payment_id" => count($payments) > 0 ? $payments[0]["id"] : null,
        ];
    }

    public function getPayment(string $paymentId)
    {
        $api = new Api($this->appKey, $this->appSecret);
        return $api->payment->fetch($paymentId);
    }
}
