<?php

namespace App\Helpers;

use App\Models\Event;
use App\Models\User;
use App\Models\RequestLog;
use App\Models\WatiRequestLog;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class Wati
{

    protected $url;

    protected $token;

    /**
     * Wati constructor.
     */
    public function __construct($token, $endPointUrl)
    {
        $this->url = $endPointUrl;
        $this->token = $token;
    }

    /**
     * @param int $page
     * @return array|null
     */
    public function getContacts($page = 1)
    {
        $url = "{$this->url}/api/v1/getContacts?pageNumber={$page}";

        $response = Http::acceptJson()->withToken($this->token)->get($url);
        if ($response->ok()) {
            return $response->json();
        }
        return null;
    }

    /**
     * @param $number
     * @param $name
     * @param array $customParams
     * @return array|null
     */
    public function createContact($number, $name, $customParams = [])
    {
        $url = "{$this->url}/api/v1/addContact/{$number}";

        $data = [
            'name' => $name,
            'customParams' => $customParams,
        ];

        // $log = RequestLog::create([
        //     "key" => $this->token,
        //     "url" => $url,
        //     "request" => $data,
        // ]);

        $response = Http::acceptJson()->withToken($this->token)->post($url, $data);

        $jsonResponse = $response->json();

        // $log->response = $jsonResponse;
        // $log->save();

        return $jsonResponse;
    }

    /**
     * @param $number
     * @param $customParams
     * @return array|null
     */
    public function updateContact($number, $customParams)
    {
        $url = "{$this->url}/api/v1/updateContactAttributes/{$number}";

        $response = Http::acceptJson()->withToken($this->token)->post($url, [
            'customParams' => $customParams,
        ]);

        if ($response->ok()) {
            return $response->json();
        }

        return null;
    }

    public function getWatiTemplates()
    {
        $url = "{$this->url}/api/v1/getMessageTemplates/";

        $broadcastTemplates = [];
        $response = Http::acceptJson()->withToken($this->token)->get($url);
        if ($response->failed()) {
            return ["status" => "failed"];
        }
        if ($response->ok()) {
            $response = $response->json();
            if (isset($response['messageTemplates'])) {
                $broadcastTemplates = $response['messageTemplates'];
            }
        }
        return ["status" => "success", "data" => $broadcastTemplates];
    }

    public function sendWatiMessage($watiId, Event $event, $broadCastName, $params, $watiTemplate = null)
    {
        $url = "{$this->url}/api/v1/sendTemplateMessage?whatsappNumber=" . $watiId;

        $templateName = $watiTemplate ?? $event->wati_template;
        $data = [
            'template_name' => $templateName,
            "broadcast_name" => $broadCastName,
            "parameters" => $params
        ];

        $log = RequestLog::create([
            "name" => "Send Wati Message",
            "data" => []
        ]);

        $response = Http::acceptJson()->withHeaders([
            'content-type' => 'text/json',
        ])->withToken($this->token)->post($url, $data);

        $log->data = $response->json();
        $log->save();
        if ($response->ok()) {
            $json = $response->json();
            Log::info("Wati {$watiId} Message Status: " . json_encode($json));
            Log::info("Wati Message end:");
            return $json;
        }
        Log::error("Wati Message Send Error: " . $watiId);
        return null;
    }

    public function sendGeneralWatiMessage($watiId, $broadCastName, $templateName, $params)
    {
        $url = "{$this->url}/api/v1/sendTemplateMessage?whatsappNumber=" . $watiId;

        $data = [
            'template_name' => $templateName,
            "broadcast_name" => $broadCastName,
            "parameters" => $params
        ];

        $response = Http::acceptJson()->withHeaders([
            'content-type' => 'text/json',
        ])->withToken($this->token)->post($url, $data);

        $log = WatiRequestLog::create([
            "key" => $this->token,
            "url" => $url,
            "request" => $data,
        ]);

        $log->response = $response->json();
        $log->save();

        // dd($response);

        if ($response->ok()) {
            $json = $response->json();
            Log::info("Wati {$watiId} Message Status: " . $json["result"]);
            return $json;
        }
        Log::error("Wati Message Send Error: " . $watiId);
        return null;
    }

    public function getApprovedTemplates($messageTemplatesData)
    {
        return collect($messageTemplatesData)
            ->where('status', 'APPROVED')
            ->values()
            ->all();
    }
}
