<?php

namespace App\Http\Controllers\API;

use App\Enums\MailTemplateCampaignEnum;
use App\Enums\MessageCampaignEnum;
use App\Enums\RoleEnum;
use App\Helpers\SendEmailCampaign;
use App\Helpers\Wati;
use App\Http\Controllers\Controller;
use App\Http\Resources\API\LoginResource;
use App\Models\AssociationUser;
use App\Models\Event;
use App\Models\EventRegistration;
use App\Models\MailTemplate;
use App\Models\MailTemplateCampaign;
use App\Models\MessageCampaign;
use App\Models\MessageTemplate;
use App\Models\RequestLog;
use App\Models\User;
use App\Rules\DisposableEmailRule;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class EventRegistrationController extends Controller
{
    public function register(Request $request)
    {
        $data = $request->all();

        $requestLog = RequestLog::create([
            'name' => 'Event Registration',
            'data' => $data
        ]);

        $request->validate([
            'event_code' => ['required', 'exists:events,code'],
            'name' => ['required'],
            'email' => ['required', 'email'],
            'contact' => ['nullable', 'integer', 'digits:10'],
            'country_code' => ['nullable'],
            'password' => ['required'],
        ]);

        try {
            $event = $this->getEvent($data['event_code']);
            $extra = $request->except('name', '_token', 'email', 'contact', 'password', 'country_code');

            $user = User::where('email', $data['email'])
                ->first();

            if ($user) {
                return $this->handleExistingUser($user, $event, $data, $extra);
            }

            // Create new user if none found
            $user = $this->createNewUser($event, $data);
            $eventRegistration = $this->handleRegisteredUser($user, $event, $extra);
            if (env('APP_ENV') == 'production') {
                $this->sendMessage($eventRegistration);
            }
            if (isset($request->api_type) && $request->api_type == "mobile") {
                return $this->sendJson(new LoginResource($user));
            }
            return $this->sendJson(['message' => 'Registration Successful.', 'user_id' => $user->id]);
        } catch (\Exception $e) {
            $this->logError($e, "Event Registration Error RequestLog ID: " . $requestLog->id);
            return $this->sendError($e->getMessage());
        }
    }

    protected function getEvent($eventCode)
    {
        return Event::where('code', $eventCode)->first();
    }

    protected function handleExistingUser(User $user, Event $event, array $data, array $extra = [])
    {
        $eventRegistration = $user->eventRegistrations()->where('event_id', $event->id)->first();

        if ($eventRegistration) {
            $eventRegistration->extra = $extra;
            $eventRegistration->save();

            if ($user->email === $data['email']) {
                return $this->sendError("Email already registered.");
            }
        }

        $user->name = $data['name'];
        if (isset($data['contact'])) {
            $user->contact = $data['contact'];
        }
        if (isset($data['password'])) {
            $user->password = Hash::make($data['password']);
        }
        $user->save();

        AssociationUser::UpdateOrCreate([
            'user_id' => $user->id,
            'association_id' => $event->association_id,
        ]);

        $eventRegistration = $this->handleRegisteredUser($user, $event, $extra);
        if (env('APP_ENV') == 'production') {
            $this->sendMessage($eventRegistration);
        }
        if (isset($data['api_type']) && $data['api_type'] == "mobile") {
            return $this->sendJson(new LoginResource($user));
        }
        return $this->sendJson(['message' => 'Registration Successful.', 'user_id'=> $user->id]);
    }

    protected function handleRegisteredUser(User $user, Event $event, array $extra = [])
    {
        return EventRegistration::create([
            'user_id' => $user->id,
            'event_id' => $event->id,
            'association_id' => $event->association_id ?? null,
            'extra' => $extra,
        ]);
    }

    protected function createNewUser(Event $event, array $data)
    {
        $user = User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'contact' => $data['contact'] ?? null,
            'pin' => $data['password'],
            'password' => Hash::make($data['password'] ?? substr($data['contact'], -4)),
            'country_code' => isset($data['country_code']) ? '+' . str_replace('+', '', $data['country_code']) : '+91',
            'role' => RoleEnum::MEMBER,
            'association_id' => $event->association_id ?? null,
        ]);

        AssociationUser::UpdateOrCreate([
            'user_id' => $user->id,
            'association_id' => $event->association_id,
        ]);

        $userDetail['member_id'] = $user->id;
        $userDetail["association_id"] = $event->association_id ?? null;
        $user->memberDetail()->create($userDetail);

        return $user;
    }

    protected function sendMessage(EventRegistration $eventRegistration)
    {
        $event = $eventRegistration->event;
        $user = $eventRegistration->user;
        if (!$user || !$event) {
            return;
        }

        // send mail
        if ($event->mailTemplate) {
            Log::info('SEND EMAIL START');
            $disposableEmailRule = new DisposableEmailRule();
            if (!$disposableEmailRule->isDisposable($user->email)) {
                $this->createMailCampaign($user, $event->mailTemplate);
            } else {
                Log::info('USER MAIL IS DISPOSABLE: ' . $user->email);
            }
            Log::info('SEND EMAIL END');
        }

        if ($event->id == 6 && $user->contact) {
            Log::info("Wati Message Start:" . $user->id);
            $watiToken = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJqdGkiOiJmZTQyZWI4ZS1iYzI1LTQ2ODQtODdlOS1lZTdjMDY0MzNjNzQiLCJ1bmlxdWVfbmFtZSI6ImluZm8udmRvdGVjaG5vbG9naWVzQGdtYWlsLmNvbSIsIm5hbWVpZCI6ImluZm8udmRvdGVjaG5vbG9naWVzQGdtYWlsLmNvbSIsImVtYWlsIjoiaW5mby52ZG90ZWNobm9sb2dpZXNAZ21haWwuY29tIiwiYXV0aF90aW1lIjoiMTAvMTYvMjAyNCAwODo1MzozNSIsInRlbmFudF9pZCI6IjEwMTQxNiIsImRiX25hbWUiOiJtdC1wcm9kLVRlbmFudHMiLCJodHRwOi8vc2NoZW1hcy5taWNyb3NvZnQuY29tL3dzLzIwMDgvMDYvaWRlbnRpdHkvY2xhaW1zL3JvbGUiOiJBRE1JTklTVFJBVE9SIiwiZXhwIjoyNTM0MDIzMDA4MDAsImlzcyI6IkNsYXJlX0FJIiwiYXVkIjoiQ2xhcmVfQUkifQ.xFYRkdcukX7fBCsP1HlknrYN9n6UXVA6SJa4D2UMjMc";
            $watiEndPointUrl = "https://live-mt-server.wati.io/101416";

            $wati = new Wati($watiToken, $watiEndPointUrl);

            $this->sendWatiMessage($wati, $user, $event);
        }

        // //send wati msg
        // if ($event->messageTemplate) {
        //     Log::info('SEND WATI MSG START');
        //     $this->createWatiCampaign($user, $event->messageTemplate);
        //     Log::info('SEND WATI MSG End');
        // }
        return;
    }

    protected function createMailCampaign(User $user, MailTemplate $mailTemplate)
    {
        $usersArray[] = $user->id;
        $mailTemplateCampaign = new MailTemplateCampaign();
        $mailTemplateCampaign->template_name = $mailTemplate->template_id;
        $mailTemplateCampaign->scheduled_date = date('Y-m-d');
        $mailTemplateCampaign->scheduled_time = date('H:i');
        $mailTemplateCampaign->total_users = count($usersArray);
        $mailTemplateCampaign->users = $usersArray;
        $mailTemplateCampaign->success_users = [];
        $mailTemplateCampaign->fail_users = [];
        $mailTemplateCampaign->status = MailTemplateCampaignEnum::PENDING;
        $mailTemplateCampaign->schedule_type = 'send_now';
        $mailTemplateCampaign->save();
        $host = request()->getHost();
        (new SendEmailCampaign())->sendEmailCampaign($host, $mailTemplateCampaign);
        return;
    }

    protected function createWatiCampaign(User $user, MessageTemplate $messageTemplate)
    {
        $usersArray[] = $user->id;
        $messageCampaign = new MessageCampaign();
        $messageCampaign->template_name = $messageTemplate->name;
        $messageCampaign->scheduled_date = date('Y-m-d');
        $messageCampaign->scheduled_time = date('H:i');
        $messageCampaign->total_users = count($usersArray);
        $messageCampaign->users = $usersArray;
        $messageCampaign->status = MessageCampaignEnum::PENDING;
        $messageCampaign->schedule_type = 'send_now';
        $messageCampaign->message_template_id = $messageTemplate->id;
        $messageCampaign->scheduled_at = date('Y-m-d H:i:s');
        $messageCampaign->save();

        $host = request()->getHost();
        $messageCampaign->start($host);
        return;
    }

    protected function sendWatiMessage(Wati $wati, User $user, Event $event)
    {
        $broadCastName = time();

        $formattedDate = date('d-m-Y', strtotime($event->date));
        $params = [
            ["name" => "name", "value" => $user->name],
            ["name" => "event_date", "value" => $formattedDate],
            ["name" => "time", "value" => "8:00 PM to 9:00 PM"],
        ];
        $this->syncUser($wati, $user);

        $wati->sendWatiMessage('91' . $user->contact, $event, $broadCastName, $params);
    }

    protected function syncUser(Wati $wati, User $user)
    {
        if (!$user->wati_id) {
            $this->createWatiContact($wati, $user);
        }

        if (Carbon::now()->addMinutes(-30)->greaterThan($user->created_at)) {
            $this->updateWatiContact($wati, $user);
        }
    }

    protected function createWatiContact(Wati $wati, User $user)
    {
        $countryCode = str_replace("+", "", $user->country_code);
        $response = $wati->createContact("{$countryCode}{$user->contact}", $user->name, []);
        Log::info("create contact" . json_encode($response));
        if ($response && isset($response['contact'])) {
            // $user->wati_wa_id = $response['contact']['wAid'];
            // $user->wati_id = $response['contact']['id'];
            // $user->save();
        } else {
            Log::error("Wati Create Error: User: {$user->id} - " . json_encode($response));
        }
    }

    protected function updateWatiContact(Wati $wati, User $user)
    {
        $countryCode = str_replace("+", "", $user->country_code);
        $response = $wati->updateContact("{$countryCode}{$user->contact}", []);
        Log::info("create contact" . json_encode($response));
        if ($response) {
            Log::error("Wati Update Error: User: {$user->id}");
            return false;
        }

        if (!isset($response['info'])) {
            Log::error("Wati Update Error: User: {$user->id} - " . json_encode($response));
            return false;
        }
        if ($response['info'] == "Contact does not exist") {
            return false;
        }

        return true;
    }
}
