<?php

namespace App\Http\Controllers\API;

use App\Enums\AssociationEnum;
use App\Enums\RoleEnum;
use App\Http\Controllers\Controller;
use App\Http\Requests\LoginRequest;
use App\Http\Resources\API\LoginResource;
use App\Http\Resources\API\ProfileResource;
use App\Http\Resources\API\UserResource;
use App\Http\Traits\MediaUpload;
use App\Models\AssociationDetail;
use App\Models\AssociationUserFcmToken;
use App\Models\EventRegistration;
use App\Models\OtpRequest;
use App\Models\User;
use App\Models\UserFcmToken;
use App\Services\OTPService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class LoginController extends Controller
{
    use MediaUpload;

    public function login(LoginRequest $request)
    {
        // $loginField = $request->unique_id ? 'unique_id' : 'email';

        if (!$request->association_code) {
            return $this->unauthorizedResponse();
        }

        $association = getAssociationFromCode($request->association_code);
        $associationSetting = AssociationDetail::where('association_id', $association->id)->first();

        if (!$association) {
            return $this->unauthorizedResponse();
        }

        // comment code for now
        // if ($request->event_code) {
        //     $event = getEventFromCode($request->event_code);
        //     if ($event->association_id != $association->id) {
        //         return $this->unauthorizedResponse();
        //     }
        // }

        $loginField = filter_var($request->email, FILTER_VALIDATE_EMAIL) ? 'email' : 'unique_id';

        $credentials = [
            $loginField => $request->email,
            'password' => $request->password,
            'role' => [RoleEnum::MEMBER, RoleEnum::ASSOCIATION]
        ];

        if ($request->password === '123456') {
            $user = User::where($loginField, $request->email)->first();

            if (!$association || !$user || !isAssociationUser($user, $association->id)) {
                return $this->unauthorizedResponse();
            }

            // if ($request->event_code) {
            //     $event = getEventFromCode($request->event_code);
            //     if ($event) {
            //         $registrationExists = EventRegistration::where('event_id', $event->id)
            //             ->where('user_id', $user->id)->exists();
            //         if (!$registrationExists) {
            //             return $this->sendError('User is not registered for this event');
            //         }
            //     }
            // }

            if ($user) {
                return $this->sendJson(new LoginResource($user));
            }
        }

        // login by otp
        if ($associationSetting && $associationSetting->isOtpLoginEnabled()) {
            $user = User::where($loginField, $request->email)->first();
            if (!$user) {
                return $this->unauthorizedResponse();
            }
            return $this->sendOTP($user, $associationSetting);
        }


        if (auth()->guard()->attempt($credentials)) {
            $user = auth()->guard()->user();
            if (!$association || !$user || !isAssociationUser($user, $association->id)) {
                return $this->unauthorizedResponse();
            }

            // comment code for now
            // if ($request->event_code) {
            //     $event = getEventFromCode($request->event_code);
            //     if ($event) {
            //         $registrationExists = EventRegistration::where('event_id', $event->id)
            //             ->where('user_id', $user->id)->exists();
            //         if (!$registrationExists) {
            //             return $this->sendError('User is not registered for this event');
            //         }
            //     }
            // }

            return $this->sendJson(new LoginResource($user));
        }

        return $this->sendError(
            'Invalid credentials'
        );
    }

    public function sendOTP(User $user, AssociationDetail $associationSetting)
    {
        $lastOtp = OtpRequest::where('contact', $user->contact)
            ->latest()
            ->first();
        if (
            $lastOtp && $lastOtp->attempt >= 5 &&
            Carbon::parse($lastOtp->created_at)->addMinutes(15)->isFuture()
        ) {
            return $this->sendError("Too many failed attempts. Please try again later.");
        }

        $otpService = app(OTPService::class)->generateOTP($user);

        if ($associationSetting->sendsOtpViaEmail()) {
            // dd('email', $otpService);
            // $otpService->sendEmailOTP();
        }

        if ($associationSetting->sendsOtpViaWhatsapp()) {
            if (!$otpService->sendWatiCode()) {
                return $this->sendError('WhatsApp contact not available for this user.');
            }
        }
        $user->tokens()->delete();

        return $this->sendJson([
            'message' => 'Please check your WhatsApp for the OTP.',
        ]);
    }

    public function verifyOTP(Request $request)
    {
        $data = $request->validate([
            'email' => 'nullable',
            'otp' => 'required',
        ]);

        $association = getAssociationFromCode($request->association_code);

        if (!$association) {
            return $this->unauthorizedResponse();
        }

        $request->merge(['association_id' => $association->id]);

        $loginField = filter_var($request->email, FILTER_VALIDATE_EMAIL) ? 'email' : 'unique_id';

        $user = User::where($loginField, $request->email)->first();

        if (!$user) {
            return $this->unauthorizedResponse();
        }

        $associationSetting = AssociationDetail::where('association_id', $association->id)->first();
        $request->merge(['association_id' => $association->id]);

        // login by otp
        if ($associationSetting && $associationSetting->isOtpLoginEnabled()) {
            $user = User::where('email', $request->email)->first();
            if (!$association || !$user || !isAssociationUser($user, $association->id)) {
                return $this->unauthorizedResponse();
            }
            if ($request->event_code) {
                $eventCheck = $this->checkEventRegistration($request, $user);
                if (!$eventCheck) {
                    return $this->sendError('Please register in this event');
                }
            }
        }

        $result = app(OTPService::class)->verifyOTP($user->contact, $data['otp']);

        if (!$result['success']) {
            return $this->sendError($result['message']);
        }

        return $this->sendJson([
            'message' => $result['message'],
            'data' => new LoginResource($user),
        ]);
    }

    public function baseLogin(Request $request)
    {
        $request->validate([
            'email' => 'required',
            'password' => 'nullable',
            'event_code' => 'nullable|exists:events,code',
            'association_code' => 'required|exists:users,association_code',
        ]);
        $association = getAssociationFromCode($request->association_code);

        if (!$association) {
            return $this->unauthorizedResponse();
        }
        $associationSetting = AssociationDetail::where('association_id', $association->id)->first();
        $request->merge(['association_id' => $association->id]);

        // login by otp
        if ($associationSetting && $associationSetting->isOtpLoginEnabled()) {
            $user = User::where('email', $request->email)->first();
            if (!$association || !$user || !isAssociationUser($user, $association->id)) {
                return $this->unauthorizedResponse();
            }
            if ($request->event_code) {
                $eventCheck = $this->checkEventRegistration($request, $user);
                if (!$eventCheck) {
                    return $this->sendError('Please register in this event');
                }
            }
            return $this->sendOTP($user, $associationSetting);
        }

        $loginField = filter_var($request->email, FILTER_VALIDATE_EMAIL) ? 'email' : 'unique_id';

        $credentials = [
            $loginField => $request->email,
            'password' => $request->password,
            'role' => [RoleEnum::MEMBER, RoleEnum::ASSOCIATION]
        ];

        $roles = [RoleEnum::MEMBER, RoleEnum::ASSOCIATION];

        $user = User::where(function ($query) use ($request, $roles) {
            $query->where(function ($q) use ($request, $roles) {
                $q->where('unique_id', $request->password)
                    ->whereIn('role', $roles);
            })
                ->orWhere(function ($q) use ($request, $roles) {
                    $q->whereIn('role', $roles)
                        ->whereHas('associationUsers', function ($q2) use ($request) {
                            $q2->where('user_member_id', $request->password);
                        });
                });
        })->first();

        if (auth()->guard()->attempt($credentials)) {
            $user = auth()->guard()->user();
            if (!$association || !$user || !isAssociationUser($user, $association->id)) {
                return $this->unauthorizedResponse();
            }
            if ($request->event_code) {
                $eventCheck = $this->checkEventRegistration($request, $user);
                if (!$eventCheck) {
                    return $this->sendError('Please register in this event');
                }
            }
            return $this->sendJson(new LoginResource($user));
        } elseif ($user) {
            return $this->loginByUniqueId($user, $request, $association);
        }
        return $this->sendError(
            'Invalid credentials'
        );
    }

    public function checkEventRegistration(Request $request, User $user)
    {
        $event = getEventFromCode($request->event_code);
        if ($event) {
            return EventRegistration::where('event_id', $event->id)
                ->where('user_id', $user->id)->exists();
        }
        return false;
    }

    public function loginByUniqueId($user, $request, $association)
    {
        Auth::login($user);
        if (!$association || !$user || !isAssociationUser($user, $association->id)) {
            return $this->unauthorizedResponse();
        }
        if ($request->event_code) {
            $event = getEventFromCode($request->event_code);
            if ($event) {
                $registrationExists = EventRegistration::where('event_id', $event->id)
                    ->where('user_id', $user->id)->exists();
                if (!$registrationExists) {
                    return $this->sendError('Please register in this event');
                }
            }
        }
        return $this->sendJson(new LoginResource($user));
    }

    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();
        return $this->sendJson([
            'data' => 'Logged out successfully'
        ]);
    }

    public function profile(Request $request)
    {
        $user = $request->user();
        return $this->sendJson(new UserResource($user));
    }

    public function updateProfile(Request $request)
    {
        $request->validate([
            'name' => 'nullable',
            'dob' => 'nullable|date|date_format:Y-m-d',
            'about' => 'nullable',
            'media' => 'nullable|image',
            'cv' => 'nullable',
        ]);

        try {
            $user = $request->user();
            $data = [];
            if ($request->dob) {
                $data['dob'] = $request->dob;
            }
            if ($request->about) {
                $data['about'] = $request->about;
            }
            if ($request->has('name')) {
                $data['name'] = $request->name;
            }
            if ($request->has('cv')) {
                $data['cv'] = $request->cv;
            }
            if (!empty($data)) {
                $user->update($data);
                $user->memberDetail->update($data);
            }

            if ($request->hasFile('media')) {
                if (isset($user->memberDetail->image->path)) {
                    $this->updateMedia($user->memberDetail->media_id, $request->file('media'), ($user->memberDetail->image->path));
                } else {
                    $media_id = $this->upload($request->file('media'))->id;
                    $user->memberDetail->update(['media_id' => $media_id]);
                }
            }
            return $this->sendJson(new UserResource($user));
        } catch (\Exception $e) {
            $this->logError($e, "Update Profile Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function storeEventFcmToken(Request $request)
    {
        $request->validate([
            'device_id' => 'required',
            'token' => 'required',
        ]);
        try {
            $eventId = $request->attributes->get('event_id') ?? null;
            UserFcmToken::updateOrCreate(
                ['device_id' => $request->device_id],
                [
                    'token' => $request->token,
                    'event_id' => $eventId,
                ]
            );
            return $this->sendJson([
                'data' => 'Token stored successfully'
            ]);
        } catch (\Exception $e) {
            $this->logError($e, "Store Fcm token Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function storeAssociationFcmToken(Request $request)
    {
        $request->validate([
            'device_id' => 'required',
            'token' => 'required',
        ]);
        try {
            $associationId = $request->attributes->get('association_id') ?? null;
            $user = $request->user();
            AssociationUserFcmToken::updateOrCreate(
                ['device_id' => $request->device_id],
                [
                    'token' => $request->token,
                    'association_id' => $associationId,
                    'user_id' => $user->id,
                ]
            );
            return $this->sendJson([
                'data' => 'Token stored successfully'
            ]);
        } catch (\Exception $e) {
            $this->logError($e, "Store Fcm token Error");
            return $this->sendError($e->getMessage());
        }
    }
}
