<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Http\Resources\API\PostCommentResource;
use App\Models\Post;
use App\Models\PostComment;
use App\Models\User;
use Illuminate\Http\Request;

class PostCommentController extends Controller
{
    public function index(Request $request, Post $post)
    {
        try {
            $postComments = $post->comments()->latest()->simplePaginate();
            return PostCommentResource::collection($postComments);
        } catch (\Exception $e) {
            $this->logError($e, "comment get");
            return $this->sendError($e->getMessage());
        }
    }

    public function store(Request $request, Post $post)
    {
        $user = $request->user();
        $validator = validator($request->all(), [
            'comment' => ['required', 'string'],
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 400);
        }

        try {
            $postComment = $post->saveComment($user, $request->comment);
            return $this->sendJson(new PostCommentResource($postComment));
        } catch (\Exception $e) {
            $this->logError($e, "store comment");
            return $this->sendError($e->getMessage());
        }
    }

    public function update(Request $request, PostComment $comment)
    {
        $validator = validator($request->all(), [
            'comment' => ['required', 'string'],
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 400);
        }
        try {
            $user = $request->user();
            if (!$comment->isOwnedByUser($user->id)) {
                return $this->sendError(['message' => 'Unauthorized']);
            }
            $postComment = $comment->post->updateComment($comment, $request->comment);
            return $this->sendJson(new PostCommentResource($postComment));
        } catch (\Exception $e) {
            $this->logError($e, "update comment");
            return $this->sendError($e->getMessage());
        }
    }

    public function delete(Request $request, PostComment $comment)
    {
        try {
            $user = $request->user();
            if (!$comment->isOwnedByUser($user->id)) {
                return $this->sendError(['message' => 'Unauthorized']);
            }
            $comment->delete();
            return $this->sendJson(['message' => 'Comment deleted successfully']);
        } catch (\Exception $e) {
            $this->logError($e, "delete comment");
            return $this->sendError($e->getMessage());
        }
    }
}
