<?php

namespace App\Http\Controllers\API;

use App\Enums\RoleEnum;
use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Resources\API\PostResource;
use App\Http\Resources\API\UserResource;
use App\Http\Traits\MediaUpload;
use App\Models\Media;
use App\Models\Post;
use App\Models\User;
use Illuminate\Http\Request;

class PostController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        try {
            $query = Post::where('association_id', $request->attributes->get('association_id'))->where('status', StatusEnum::ACTIVE)->orderByDesc('id');

            if ($userId = $request->get('user_id', null)) {
                $query->where('user_id', $userId);
            }
            $posts = $query->simplePaginate(25);
            return PostResource::collection($posts);
        } catch (\Exception $e) {
            $this->logError($e, "Get Posts Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function store(Request $request)
    {
        $data = $request->validate(
            [
                'description' => 'required',
                'link' => 'nullable|url',
                "medias" => 'nullable|array',
                "medias.*" => 'nullable|file|image|max:20048',
            ],
            [
                "medias.*.image" => 'Each file must be an image.'
            ]
        );
        try {
            $user = $request->user();
            $post = new Post($data);
            $post->status = StatusEnum::ACTIVE;
            $post->user_id = $this->getCurrentUserId();
            $post->association_id = $user->role == RoleEnum::ASSOCIATION ? $user->id : $user->association_id;
            $post->save();

            if ($request->hasFile('medias')) {
                foreach ($request->file('medias') as $files) {
                    $media = $this->upload($files);
                    $media->post_id = $post->id;
                    $media->save();
                }
            }
            return $this->sendJson(['data' => new PostResource($post)]);
        } catch (\Exception $e) {
            $this->logError($e, "Store Posts Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function like(Request $request, Post $post)
    {
        try {
            $user = $request->user();
            if ($post->isLiked($user)) {
                $post->removeLike($user);
                $message = 'Post Unliked';
            } else {
                $post->like($user);
                $message = 'Post Liked';
            }

            return $this->sendJson([
                'message' => $message
            ]);
        } catch (\Exception $e) {
            $this->logError($e, "like post Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function getLikes(Post $post)
    {
        $users = $post->likedByUsers()->orderBy("post_likes.created_at", "desc")->simplePaginate(25);
        return UserResource::collection($users);
    }

    public function bookmark(Request $request, Post $post)
    {
        try {
            $user = $request->user();
            if ($post->isBookmarked($user)) {
                $post->removeBookmark($user);
                $message = 'Post remove from bookmark';
            } else {
                $post->bookmarkPost($user);
                $message = 'Post Bookmarked';
            }

            return response()->json([
                'message' => $message
            ]);
        } catch (\Exception $e) {
            $this->logError($e, "bookmark Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function userBookmarkedPost(Request $request)
    {
        try {
            $user = $request->user();
            $savedPost = $user->bookmarkedPosts()->latest()->simplePaginate();
            return PostResource::collection($savedPost);
        } catch (\Exception $e) {
            $this->logError($e, "User bookmark post");
            return $this->sendError($e->getMessage());
        }
    }

    public function update(Request $request, Post $post)
    {
        $data = $request->validate([
            'title' => ['required', 'max:255'],
            'description' => ['required'],
            'link' => ['required', 'url'],
            'medias' => 'nullable|array',
        ]);
        try {
            $user = $request->user();
            if ($user->id != $post->user_id) {
                return $this->sendError("Unauthorized!");
            }

            $post->update($data);
            if ($request->hasFile('medias')) {
                $post->medias()->delete();
                foreach ($request->file('medias') as $file) {
                    $media = $this->upload($file);
                    $media->post_id = $post->id;
                    $media->save();
                }
            }

            return $this->sendJson(['data' => new PostResource($post)]);
        } catch (\Exception $e) {
            $this->logError($e, "Update Posts Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function delete(Request $request, Post $post)
    {
        $user = $request->user();
        if ($post->user_id == $user->id) {
            $post->delete();
            return $this->sendJson([
                "message" => "Your post has been deleted!"
            ]);
        }
        return $this->sendError("Unauthorized!");
    }
}
