<?php

namespace App\Http\Controllers\Admin;

use App\Enums\RoleEnum;
use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Models\Committee;
use App\Models\Country;
use App\Models\Event;
use App\Models\MailTemplate;
use App\Models\MemberRole;
use App\Models\MessageTemplate;
use App\Models\User;
use App\Rules\DisposableEmailRule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

class AssociationController extends Controller
{
    public function index(Request $request)
    {
        $associations = User::where('role', RoleEnum::ASSOCIATION)->orderBy('id', 'desc');

        if ($request->nameAndMobile) {
            $associations->where(function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->nameAndMobile . '%')
                    ->orWhere('contact', 'like', '%' . $request->nameAndMobile . '%');
            });
        }
        if ($request->email) {
            $associations->where("email", 'like', '%' . $request->email . '%');
        }
        if ($request->association_code) {
            $associations->where("association_code", 'like', '%' . $request->association_code . '%');
        }

        if ($request->export == 'export') {
            $associations = $associations->get();
            $filename = 'association.csv';
            $columns = [
                'Sr No.',
                'Name',
                'Email',
                'Contact',
                'Date',
            ];

            return exportToCsv($request, $associations, $filename, $columns, function ($associations, $index) {
                return [
                    $index,
                    $associations->name ?? '-',
                    $associations->email ?? '-',
                    $associations->contact ?? '-',
                    $associations->created_at ? $associations->created_at->timezone('Asia/Kolkata')->format('d-m-Y | h:i A') : '',
                ];
            });
        }

        $associations = $associations->paginate(25);

        return view('admin.associations.index', compact('associations', 'request'));
    }

    public function create()
    {
        $countryCodes = Country::get();
        return view('admin.associations.add', compact('countryCodes'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required'],
            'email' => ['required', 'unique:users,email', new DisposableEmailRule()],
            'password' => ['required'],
            'country_code' => ['required'],
            'contact' => ['required', 'integer'],
            'status' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);
        $data['pin'] = $data['password'];
        $data['password'] = Hash::make($data['password']);
        $data['country_code'] = '+' . str_replace('+', '', $data['country_code']);
        $data['role'] = RoleEnum::ASSOCIATION;
        $data['association_code'] = $this->generateUniqueCode();
        $association = User::create($data);

        $association->update([
            'association_id' => $association->id
        ]);

        $roleData = [
            [
                "association_id" => $association->id,
                "name" => "President",
                "status" => StatusEnum::ACTIVE,
                'created_at' => now()
            ],
            [
                "association_id" => $association->id,
                "name" => "Secretary",
                "status" => StatusEnum::ACTIVE,
                'created_at' => now()
            ],
        ];
        MemberRole::insert($roleData);

        $committeedata = [
            [
                "association_id" => $association->id,
                "name" => "Executive",
                "status" => StatusEnum::ACTIVE,
                'created_at' => now()
            ],
            [
                "association_id" => $association->id,
                "name" => "Standing",
                "status" => StatusEnum::ACTIVE,
                'created_at' => now()
            ],
        ];
        Committee::insert($committeedata);

        return redirect()->route('associations.index')->with('success', 'Association Created');
    }

    public function edit(User $association)
    {
        $countryCodes = Country::get();
        $code = $association->country_code;
        $selectedCountryCode = ltrim($code, '+');
        return view('admin.associations.edit', compact('association', 'countryCodes', 'selectedCountryCode'));
    }

    public function update(User $association, Request $request)
    {
        $request->validate([
            'name' => ['required'],
            'email' => [
                'required',
                Rule::unique('users')->ignore($association->id),
                new DisposableEmailRule()
            ],
            'country_code' => ['required'],
            'contact' => ['required', 'integer'],
            'status' => ['required'],
        ]);

        $data = $request->except(['_token', '_method']);
        $data['country_code'] = '+' . str_replace('+', '', $data['country_code']);
        if ($data['password']) {
            $data['pin'] = $data['password'];
            $data['password'] = Hash::make($data['password']);
        }

        if (!$association->association_code) {
            $data['association_code'] = $this->generateUniqueCode();
        }

        $association->update($data);
        return redirect()->route('associations.index')->with('success', 'Association Updated');
    }

    public function statusChange(User $association)
    {
        if ($association->status == StatusEnum::ACTIVE) {
            $association->status = StatusEnum::INACTIVE;
        } else {
            $association->status = StatusEnum::ACTIVE;
        }
        $association->save();
        return response()->json(["status" => 200, "message" => "Status Changed"]);
    }

    public function destroy(User $association)
    {
        $association->delete();
        return response()->json(["status" => 200, "message" => "Association Deleted"]);
    }

    private function generateUniqueCode()
    {
        do {
            $code = strtoupper(Str::random(6));
        } while (User::where('association_code', $code)->exists());

        return $code;
    }

    public function events(User $association, Request $request)
    {
        $events = Event::latest()->where('association_id', $association->id);
        $events = $events->filterEvent();
        $events = $events->paginate(20);
        return view('admin.associations.events', compact('association', 'events', 'request'));
    }

    public function participants(Event $event, Request $request)
    {

        $eventParticipants = $event->eventRegistrations();
        if ($request->filter) {
            $eventParticipants->whereHas('user', function ($q) use ($request) {
                $q->where(function ($query) use ($request) {
                    $query->where('name', 'like', '%' . $request->filter . '%')
                        ->orWhere('contact', 'like', '%' . $request->filter . '%');
                });
            });
        }
        if ($request->email) {
            $eventParticipants->whereHas('user', function ($q) use ($request) {
                $q->where('email', $request->email);
            });
        }

        if ($request->export == 'export') {
            $eventParticipants = $eventParticipants->get();
            $filename = $event->title . "_event-participants.csv";
            $columns = [
                'Sr No.',
                'Name',
                'Email',
                'Contact',
                'Date',
            ];

            return exportToCsv($request, $eventParticipants, $filename, $columns, function ($eventParticipants, $index) {
                return [
                    $index,
                    $eventParticipants->user->name ?? '-',
                    $eventParticipants->user->email ?? '-',
                    $eventParticipants->user->contact ?? '-',
                    $eventParticipants->user->created_at ? $eventParticipants->user->created_at->timezone('Asia/Kolkata')->format('d-m-Y | h:i A') : '',
                ];
            });
        }
        $eventParticipants = $eventParticipants->paginate(20);
        $messageTemplates = MessageTemplate::where('status', MessageTemplate::APPROVED)->select('id', 'name')->get();
        $mailTemplates = MailTemplate::where('status', StatusEnum::ACTIVE)->select('id', 'name')->get();
        return view('admin.associations.participant.index', compact('event', 'eventParticipants', 'request', 'messageTemplates', 'mailTemplates'));
    }
}
