<?php

namespace App\Http\Controllers\Association;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Calendar;
use Illuminate\Http\Request;

class CalendarController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $calendars = Calendar::where('association_id', $this->getCurrentUserId())
            ->orderByDesc('date');
        $calendars = $calendars->filterCalendars();

        $calendars = $calendars->paginate(25);
        return view('association.calendar.index', compact('calendars', 'request'));
    }

    public function create()
    {
        return view('association.calendar.add');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => ['required', 'string'],
            'image' => ['required', 'image'],
            'date' => ['required'],
            'link' => ['required', 'url'],
            'description' => ['required', 'string'],
        ]);
        $data = $request->except(['_token', '_method', 'image']);
        if (isset($request->image)) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $data["date"] = date('Y-m-d', strtotime($request->date));
        $data["association_id"] = $this->getCurrentUserId();
        Calendar::create($data);
        return redirect()->route('calendars.index')->with('success', 'Calendar data Created Successfully.');
    }

    public function edit(Calendar $calendar)
    {
        return view('association.calendar.edit', compact('calendar'));
    }

    public function update(Calendar $calendar, Request $request)
    {
        $request->validate([
            'title' => ['required', 'string'],
            'banner' => ['nullable', 'image'],
            'date' => ['required'],
            'link' => ['required', 'url'],
            'description' => ['required', 'string'],
        ]);

        $data = $request->except(['_token', '_method']);
        if (isset($request->image)) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $data["date"] = date('Y-m-d', strtotime($request->date));
        $calendar->update($data);
        return redirect()->route('calendars.index')->with('success', 'Calendar data Updated Successfully.');
    }

    public function statusChange(Calendar $calendar)
    {
        $newStatus = ($calendar->status == StatusEnum::ACTIVE) ? StatusEnum::INACTIVE : StatusEnum::ACTIVE;
        $calendar->status = $newStatus;
        $calendar->save();
        return response()->json(["status" => 200, "message" => "Status Changed Successfully."]);
    }

    public function destroy(Calendar $calendar)
    {
        $calendar->delete();
        return response()->json(["status" => 200, "message" => "Calendar Deleted Successfully."]);
    }
}
