<?php

namespace App\Http\Controllers\Association;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\InternationalAffiliation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class InternationalAffiliationController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $internationalAffiliations = InternationalAffiliation::where('association_id', Auth::user()->id);
        if (!empty($request->name) && $request->name != 'all') {
            $internationalAffiliations->where('name', 'like', "%" . $request->name . "%");
        }

        if (!empty($request->status) && $request->status != 'all') {
            if ($request->status == "inactive") {
                $statusData = StatusEnum::INACTIVE;
            }
            if ($request->status == "active") {
                $statusData = StatusEnum::ACTIVE;
            }
            $internationalAffiliations->where('status', $statusData);
        }
        $internationalAffiliations = $internationalAffiliations->orderBy('index')->paginate(25);


        return view('association.international-affiliations.index', compact('internationalAffiliations', 'request'));
    }

    public function create()
    {
        return view('association.international-affiliations.add');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required'],
            'link' => ['nullable', 'url'],
            'image' => ['required', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);

        if (isset($request->image)) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $data["association_id"] = Auth::user()->id;
        $data["index"] = $request->index ?? InternationalAffiliation::max('index') + 1;
        $data["index"] = (int) $data["index"];
        InternationalAffiliation::create($data);

        return redirect()->route('international.affiliations.index')->with('success', 'International Affiliations created');
    }

    public function edit(InternationalAffiliation $internationalAffiliation)
    {
        return view('association.international-affiliations.edit', compact('internationalAffiliation'));
    }

    public function update(InternationalAffiliation $internationalAffiliation, Request $request)
    {
        $request->validate([
            'name' => ['required'],
            'link' => ['nullable', 'url'],
            'image' => ['nullable', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);
        if (isset($request->image)) {
            $this->updateMedia($internationalAffiliation->media_id, $request->image, $internationalAffiliation->image->path);
        }

        $data["index"] = $request->index ?? InternationalAffiliation::max('index') + 1;
        $data["index"] = (int) $data["index"];
        $internationalAffiliation->update($data);
        return redirect()->route('international.affiliations.index')->with('success', 'International Affiliations Updated');
    }

    public function statusChange(InternationalAffiliation $internationalAffiliation)
    {
        $newStatus = ($internationalAffiliation->status == StatusEnum::ACTIVE) ? StatusEnum::INACTIVE : StatusEnum::ACTIVE;
        $internationalAffiliation->status = $newStatus;
        $internationalAffiliation->save();
        return response()->json(["status" => 200, "message" => "Status Changed Successfully."]);
    }

    public function destroy(InternationalAffiliation $internationalAffiliation)
    {
        $internationalAffiliation->delete();
        return response()->json(["status" => 200, "message" => "International Affiliations Deleted Successfully."]);
    }

    public function updateIndex(Request $request, InternationalAffiliation $internationalAffiliation)
    {
        $request->validate([
            'index' => 'required',
        ]);
        try {
            $internationalAffiliation->index = $request->index;
            $internationalAffiliation->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('international.affiliations.index');
        } catch (\Exception $e) {
            $this->logError($e, "Update index international affiliations");
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }
}
