<?php

namespace App\Http\Controllers\Association;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Notice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class NoticeController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $notices = Notice::where('association_id', Auth::user()->id)->orderBy('index');

        $notices = $notices->paginate(25);

        return view('association.notices.index', compact('notices', 'request'));
    }

    public function create()
    {
        return view('association.notices.add');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => ['required'],
            'notice_image' => ['required', 'image', 'max:2048'],
        ]);

        $data = $request->except(['_token', '_method']);

        if (isset($request->notice_image)) {
            $data['media_id'] = $this->upload($request->notice_image)->id;
        }
        $data["association_id"] = Auth::user()->id;
        $data["index"] = $request->index ?? Notice::max('index') + 1;
        $data["index"] = (int) $data["index"];
        Notice::create($data);

        return redirect()->route('notices.index')->with('success', 'Notice Created');
    }

    public function edit(Notice $notice)
    {
        return view('association.notices.edit', compact('notice'));
    }

    public function update(Notice $notice, Request $request)
    {
        $request->validate([
            'title' => ['required'],
            'notice_image' => ['nullable', 'image', 'max:2048'],
        ]);

        $data = $request->except(['_token', '_method']);
        if (isset($request->notice_image)) {
            $this->updateMedia($notice->media_id, $request->notice_image, $notice->image->path);
        }
        $data["index"] = $request->index ?? Notice::max('index') + 1;
        $data["index"] = (int) $data["index"];
        $notice->update($data);
        return redirect()->route('notices.index')->with('success', 'Notice Updated');
    }

    public function statusChange(Notice $notice)
    {
        if ($notice->status == StatusEnum::ACTIVE) {
            $notice->status = StatusEnum::INACTIVE;
        } else {
            $notice->status = StatusEnum::ACTIVE;
        }
        $notice->save();
        return response()->json(["status" => 200, "message" => "Notice Status Changed"]);
    }

    public function destroy(Notice $notice)
    {
        $notice->delete();
        return response()->json(["status" => 200, "message" => "Notice Deleted"]);
    }
}
