<?php

namespace App\Http\Controllers\Association;

use App\Enums\PollTypeEnum;
use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Models\Poll;
use Illuminate\Http\Request;

class PollController extends Controller
{
    public function index(Request $request)
    {
        $polls = Poll::where('association_id', $this->getCurrentUserId());
        $polls = $polls->paginate(25);
        return view('association.polls.index', compact('request', 'polls'));
    }

    public function create()
    {
        return view('association.polls.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'question' => 'required|string',
            'start_date' => 'required',
            'start_time' => 'required|date_format:H:i',
            'expire_date' => 'required',
            'expire_time' => 'required|date_format:H:i',
        ]);

        $data['association_id'] = $this->getCurrentUserId();
        $data['expire_date'] = date('Y-m-d', strtotime($request->expire_date));
        $data['start_date'] = date('Y-m-d', strtotime($request->start_date));
        Poll::create($data);

        return redirect()->route('polls.index')->with('success', 'Poll Created');
    }

    public function edit(Poll $poll)
    {
        return view('association.polls.edit', compact('poll'));
    }

    public function update(Request $request, Poll $poll)
    {
        $data = $request->validate([
            'question' => 'required|string',
            'start_date' => 'required',
            'start_time' => 'required|date_format:H:i',
            'expire_date' => 'required',
            'expire_time' => 'required|date_format:H:i',
        ]);

        $data['expire_date'] = date('Y-m-d', strtotime($request->expire_date));
        $data['start_date'] = date('Y-m-d', strtotime($request->start_date));

        $poll->update($data);
        return redirect()->route('polls.index')->with('success', 'Poll updated successfully');
    }

    public function statusChange(Poll $poll)
    {
        $newStatus = ($poll->status == StatusEnum::ACTIVE) ? StatusEnum::INACTIVE : StatusEnum::ACTIVE;
        $poll->status = $newStatus;
        $poll->save();
        return response()->json(["status" => 200, "message" => "Poll Status Changed"]);
    }

    public function typeChange(Poll $poll)
    {
        $newType = ($poll->declare == PollTypeEnum::DECLARE) ? PollTypeEnum::NON_DECLARE : PollTypeEnum::DECLARE;
        $poll->declare = $newType;
        $poll->save();
        return response()->json(["status" => 200, "message" => "Poll Type Changed"]);
    }

    public function destroy(Poll $poll)
    {
        $poll->delete();
        return response()->json(["status" => 200, "message" => "Poll Deleted"]);
    }
}
