<?php

namespace App\Http\Controllers\Event;

use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Event;
use App\Models\EventActivity;
use App\Models\Media;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class EventActivityController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $eventId = Session::get('event_id');
        $activites = EventActivity::where('event_id', $eventId);

        if ($request->filter) {
            $activites->where(function ($q) use ($request) {
                $q->where('name', 'like', "%{$request->filter}%")
                    ->orWhereLike('title', $request->filter);
            });
        }
        $activites = $activites->paginate(25);
        return view('event.activity.index', compact('activites', 'request'));
    }

    public function create(Request $request)
    {
        return view('event.activity.add');
    }

    public function store(Request $request, Event $event)
    {
        $request->validate([
            'title' => ['required', 'string'],
            'name' => ['nullable', 'string'],
            'link' => ['nullable', 'url'],
            'index' => ['nullable', 'integer'],
            'images' => ['required', 'array', 'min:1', 'max:10'],
            'images.*' => ['image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);

        if (isset($request->images)) {
            foreach ($request->images as $image) {
                $mediaId = $this->upload($image)->id;
                $mediaIds[] = $mediaId;
            }
            $mediaIds = array_unique($mediaIds);
        }
        $data['media_ids'] = json_encode($mediaIds);
        $data["index"] = $request->index ?? 0;
        $data["event_id"] = $event->id;
        EventActivity::create($data);
        return redirect()->route('event.activities.index')->with('success', 'Activity Created');
    }

    public function edit(EventActivity $eventActivity)
    {
        $images = collect();

        if ($eventActivity && !empty($eventActivity->media_ids)) {
            $mediaIds = json_decode($eventActivity->media_ids);
            $images = Media::whereIn('id', $mediaIds)->get();
        }

        return view('event.activity.edit', compact('eventActivity', 'images'));
    }

    public function update(EventActivity $eventActivity, Request $request)
    {
        $mediaIds = json_decode($eventActivity->media_ids);
        if (empty($mediaIds) && $request->images == null) {
            return redirect()->back()->with('error', 'Minimum One Image is Required.')->withInput();
        }
        $imageCount = count(json_decode($eventActivity->media_ids));
        $imageCount = 10 - $imageCount;
        $request->validate([
            'title' => ['required', 'string'],
            'name' => ['nullable', 'string'],
            'link' => ['nullable', 'url'],
            'index' => ['nullable', 'integer'],
            'images' => ['nullable', 'array', "max:$imageCount"],
            'images.*' => ['image', 'max:2048'],
        ]);

        $data = $request->except(['_token', '_method']);
        if (isset($request->images)) {
            foreach ($request->images as $image) {
                $mediaId = $this->upload($image)->id;
                $mediaIds[] = $mediaId;
            }
            $mediaIds = array_unique($mediaIds);
        }
        $data['media_ids'] = json_encode($mediaIds);
        $data["index"] = $request->index ?? 0;
        $eventActivity->update($data);
        return redirect()->route('event.activities.index')->with('success', 'Speaker Updated');
    }

    public function updateIndex(Request $request, EventActivity $eventActivity)
    {
        $request->validate([
            'index' => 'required',
        ]);

        try {
            $eventActivity->index = $request->index;
            $eventActivity->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('event.activities.index');
        } catch (\Exception $e) {
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }

    public function mediaDelete(Media $media, EventActivity $eventActivity)
    {
        $json = json_decode($eventActivity->media_ids);
        $updatedJson = array_filter($json, function ($value) use ($media) {
            return $value !== $media->id;
        });

        $updatedJson = array_values($updatedJson);

        $eventActivity->update([
            'media_ids' => json_encode($updatedJson),
        ]);

        $this->remove($media->path);
        $media->delete();
        return response()->json(["status" => 200, "message" => "Image Deleted"]);
    }

    public function destroy(EventActivity $eventActivity)
    {
        $eventActivity->delete();
        return response()->json(["status" => 200, "message" => "Speaker Deleted"]);
    }

}
