<?php

namespace App\Http\Controllers\Event\EventCoordinator\EventForm;

use App\Enums\EventFormFieldEnum;
use App\Http\Controllers\Controller;
use App\Models\Event;
use App\Models\EventFormWorkshop;
use App\Models\EventRegistration;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Auth;

class ConfirmRegistrationController extends Controller
{
    public function index(Request $request)
    {
        $canEdit = checkEventManagerPrivileges();
        $event = Event::find(Session::get('event_id'));
        $eventParticipants = $event->eventRegistrations();

        if ($request->filter) {
            $eventParticipants->whereHas('user', function ($q) use ($request) {
                $q->where(function ($query) use ($request) {
                    $query->where('name', 'like', '%' . $request->filter . '%')
                        ->orWhere('contact', 'like', '%' . $request->filter . '%');
                });
            });
        }
        if ($request->email) {
            $eventParticipants->whereHas('user', function ($q) use ($request) {
                $q->where('email', $request->email);
            });
        }

        if ($request->export == 'export') {
            $eventParticipants = $eventParticipants->get();
            $fileName = $event->title . "_event-participants_" . now()->timezone('Asia/Kolkata')->format('d-m-Y_H-i-s') . ".csv";
            $headers = array(
                "Content-type" => "text/csv",
                "Content-Disposition" => "attachment; filename=$fileName",
                "Pragma" => "no-cache",
                "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
                "Expires" => "0"
            );
            $columns = array(
                'Sr No.',
                'Name',
                'Email',
                'Contact',
                'Date',
            );

            $basicFields = getBasicFields();
            if ($event->eventFormInputFields) {
                foreach ($event->eventFormInputFields as $inputFormFields) {
                    if (!in_array($inputFormFields->name, $basicFields)) {
                        $columns[] = $inputFormFields->display_label;
                    }
                }
            }

            $callback = function () use ($eventParticipants, $columns, $event, $basicFields) {
                $file = fopen('php://output', 'w');
                fputcsv($file, $columns);
                $i = 1;
                foreach ($eventParticipants as $eventParticipant) {
                    $row['sr_no'] = $i;
                    $row['name'] = $eventParticipant->user->name ?? '-';
                    $row['email'] = $eventParticipant->user->email ?? '-';
                    $row['contact'] = $eventParticipant->user->contact ?? '-';
                    $row['date'] = $eventParticipant->user->created_at ? $eventParticipant->user->created_at->timezone('Asia/Kolkata')->format('d-m-Y | h:i A') : '';

                    $requiredArray = array($row['sr_no'], $row['name'], $row['email'], $row['contact'], $row['date']);
                    $extraArray = [];
                    if ($event->eventFormInputFields) {
                        foreach ($event->eventFormInputFields as $inputFormFields) {
                            if (!in_array($inputFormFields->name, $basicFields)) {
                                $extraColumn = "";
                                isset($eventParticipant) ? (isset($eventParticipant->extra[$inputFormFields['name']]) ? $extraColumn = $eventParticipant->extra[$inputFormFields['name']] : '-') : '-';
                                if (isset($extraColumn)) {
                                    $row['field'] = "";
                                    if ($inputFormFields->type == 'checkbox') {
                                        if (is_array($extraColumn)) {
                                            if (count($extraColumn) > 0) {
                                                foreach ($extraColumn as $key => $extraField) {
                                                    $row['field'] .= $extraField;
                                                    if (count($extraColumn) != $key + 1) {
                                                        $row['field'] .= ",";
                                                    }
                                                }
                                            }
                                        }
                                    } elseif ($inputFormFields->name == 'category') {
                                        $row['field'] = ($eventParticipant->extra['category_data']['category_name'] ?? '-') .
                                            " \n " .
                                            ($eventParticipant->extra['category_data']['category_amount'] ?? '-');
                                    } elseif ($inputFormFields->name == 'workshop') {
                                        $row['field'] = ($eventParticipant->extra['workshop_data']['workshop_name'] ?? $eventParticipant->extra[$inputFormFields['name']] ?? '-') .
                                            " \n " .
                                            ($eventParticipant->extra['workshop_data']['workshop_amount'] ?? '-');
                                    } elseif ($inputFormFields->name == 'register_accompany') {
                                        $row['field'] = ($eventParticipant->extra['register_accompany'] ?? '-') .
                                            " \n " .
                                            ($eventParticipant->extra['accompany_data']['accompany_amount'] ?? '');
                                    } elseif ($inputFormFields->name == 'accompany_person') {
                                        $names = '';
                                        if (isset($eventParticipant->extra['accompany_name']) && is_array($eventParticipant->extra['accompany_name'])) {
                                            foreach ($eventParticipant->extra['accompany_name'] as $data) {
                                                $names .= $data . "\n";
                                            }
                                        } else {
                                            $names = '-';
                                        }

                                        $row['field'] =
                                            ($eventParticipant->extra['accompany_person'] ?? '-') .
                                            "\n" .
                                            $names;
                                    } elseif ($inputFormFields->type == 'file_input') {
                                        $row['field'] = '-';
                                    } else {
                                        $row['field'] = $extraColumn;
                                    }
                                    $extraArray[] = $row['field'];
                                }
                            }
                        }
                    }
                    $allArray = array_merge($requiredArray, $extraArray);
                    fputcsv($file, $allArray);
                    $i++;
                }
                fclose($file);
            };
            return response()->stream($callback, 200, $headers);
        }
        $eventParticipants = $eventParticipants->paginate(20);
        $basicFields = getBasicFields();
        return view('association.event-coordinator.event-form.confirm-registration.index', compact('event', 'eventParticipants', 'request', 'basicFields', 'canEdit'));
    }

    public function destroy(EventRegistration $eventRegistration)
    {
        if (!checkEventManagerPrivileges()) {
            return $this->sendJson(["status" => 403, "message" => "You don't have permission to perform this action."], 403);
        }
        if (isset($eventRegistration->extra['workshop'])) {
            $workshop = EventFormWorkshop::where('id', $eventRegistration->extra['workshop'])
                ->first();
            if ($workshop && $workshop->no_of_registration != 0) {
                $workshop->no_of_registration -= 1;
                $workshop->update();
            }
        }
        $eventRegistration->delete();
        return $this->sendJson(["status" => 200, "message" => "User Unregister Successfully"]);
    }

    public function getData(Request $request, EventRegistration $eventRegistration)
    {
        if (!checkEventManagerPrivileges()) {
            return $this->sendJson(["status" => 403, "message" => "You don't have permission to perform this action."], 403);
        }
        try {
            $user = $eventRegistration->user;
            $event = getEventFromId(Session::get('event_id'));
            $eventFormInputFields = $event->eventFormInputFields()
                ->where('visible', EventFormFieldEnum::VISIBLE)
                ->where('label', $request->column)
                ->orWhere('name', $request->column)
                ->first();
            $fieldHtml = view('association.event-coordinator.event-form.confirm-registration.edit-field-model-body', compact('eventFormInputFields', 'eventRegistration'))->render();
            return $this->sendJson(["status" => 200, "message" => "Data get Successfully", 'data' => $fieldHtml]);
        } catch (\Exception $e) {
            $this->logError($e, "Get data for edit field Error");
            return $this->sendError($e->getMessage());
        }
    }

    public function updateFieldValue(Request $request)
    {
        if (!checkEventManagerPrivileges()) {
            return $this->sendJson(["status" => 403, "message" => "You don't have permission to perform this action."], 403);
        }

        try {
            $data = $request->all();
            if (!isset($data['key']) && $data['key'] == null) {
                return $this->sendError('Something went wrong, Please try again latter');
            }
            $eventRegistration = EventRegistration::find($request['id']);
            $extra = $eventRegistration->extra ?? [];
            $extra[$data['key']] = $data[$data['key']];
            $eventRegistration->extra = $extra;
            $eventRegistration->save();
            return redirect()->route('event.forms.confirm.registrations.index')->with('success', 'Field Updated Successfully');
        } catch (\Exception $e) {
            $this->logError($e, "Update form field error");
            return $this->sendError($e->getMessage());
        }
    }

    public function getUserData(Request $request, User $user)
    {
        if (!checkEventManagerPrivileges()) {
            return $this->sendJson(["status" => 403, "message" => "You don't have permission to perform this action."], 403);
        }

        try {
            $registrationDetails = EventRegistration::find($request->id);
            $fieldHtml = view('association.event-coordinator.event-form.confirm-registration.edit-user-model-body', compact('user', 'registrationDetails'))->render();
            return $this->sendJson(["status" => 200, "message" => "Data get Successfully", 'data' => $fieldHtml]);
        } catch (\Exception $e) {
            $this->logError($e, "Get user details error");
            return $this->sendError($e->getMessage());
        }
    }

    public function updateUserData(Request $request)
    {
        if (!checkEventManagerPrivileges()) {
            return $this->sendJson(["status" => 403, "message" => "You don't have permission to perform this action."], 403);
        }

        try {
            $data = $request->all();
            $user = User::find($data['id']);
            $user->fill($data);
            $user->save();

            // Update 
            $registrationDetails = EventRegistration::find($request->eventRegistrationId);
            $extra = $registrationDetails->extra ?? [];
            $extra['country'] = $data['country'];
            $extra['state'] = $data['state'];
            $extra['city'] = $data['city'];
            $registrationDetails->extra = $extra;
            $registrationDetails->save();

            return redirect()->route('event.forms.confirm.registrations.index')->with('success', 'User details updated successfully');
        } catch (\Exception $e) {
            $this->logError($e, "Update user details error");
            return $this->sendError($e->getMessage());
        }
    }
}
