<?php

namespace App\Http\Controllers\Event\EventCoordinator\EventForm;

use App\Enums\EventRegistrationTypeEnum;
use App\Enums\MailTemplateCampaignEnum;
use App\Helpers\SendEmailCampaign;
use App\Helpers\WatiMessage;
use App\Http\Controllers\Controller;
use App\Models\EventFormWorkshop;
use App\Models\EventMailTemplate;
use App\Models\EventRegistration;
use App\Models\MailTemplateCampaign;
use App\Models\PaymentOrder;
use App\Models\User;
use App\Rules\DisposableEmailRule;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;

class PendingRegistrationController extends Controller
{
    public function index()
    {
        $canEdit = checkEventManagerPrivileges();
        $pendingRegistrations = PaymentOrder::where('status', 'PENDING')
            ->where('payment_mode', EventRegistrationTypeEnum::OFFLINE)
            ->where('event_id', Session::get('event_id'));
        $pendingRegistrations = $pendingRegistrations->paginate(25);
        return view('association.event-coordinator.event-form.pending-registration.index', compact('pendingRegistrations', 'canEdit'));
    }

    public function confirmationRegistration(PaymentOrder $paymentOrder)
    {
        if (!checkEventManagerPrivileges()) {
            return response()->json(["status" => 403, "message" => "You don't have permission to perform this action"]);
        }
        if (isset($paymentOrder->member_data['workshop'])) {
            $workshop = EventFormWorkshop::where('id', $paymentOrder->member_data['workshop'])
                ->where('event_id', Session::get('event_id'))
                ->first();
            if ($workshop) {
                $workshop->no_of_registration += 1;
                $workshop->update();
            }
        }

        $eventRegistration = EventRegistration::create([
            'event_id' => $paymentOrder->event_id,
            'user_id' => $paymentOrder->user_id,
            'association_id' => $paymentOrder->event->association_id,
            'extra' => $paymentOrder->member_data,
            'paid_amount' => $paymentOrder->amount,
            'payment_info' => EventRegistrationTypeEnum::OFFLINE_REGISTRATION,
        ]);

        $paymentOrder->status = "SUCCESS";
        $paymentOrder->save();

        if (env('APP_ENV') == 'production') {
            $this->sendMessage($eventRegistration);
        }

        return response()->json(["status" => 200, "message" => "Registration confirmed"]);
    }

    protected function sendMessage(EventRegistration $eventRegistration)
    {
        $event = $eventRegistration->event;
        $user = $eventRegistration->user;
        if (!$user || !$event) {
            return;
        }

        // send mail
        if ($event->eventConfig && $event->eventConfig->offlineMailTemplateActive) {
            Log::info('SEND EMAIL START');
            $disposableEmailRule = new DisposableEmailRule();
            if (!$disposableEmailRule->isDisposable($user->email)) {
                $this->createMailCampaign($user, $event->eventConfig->offlineMailTemplateActive);
            } else {
                Log::info('USER MAIL IS DISPOSABLE: ' . $user->email);
            }
            Log::info('SEND EMAIL END');
        }
        if ($event->eventConfig && $event->eventConfig->offline_wati_template) {
            WatiMessage::sendMessageIfRequired($user, $event, $event->eventConfig->offline_wati_template);
        }
        return;
    }

    protected function createMailCampaign(User $user, EventMailTemplate $mailTemplate)
    {
        $usersArray[] = $user->id;
        $mailTemplateCampaign = new MailTemplateCampaign();
        $mailTemplateCampaign->template_name = $mailTemplate->id;
        $mailTemplateCampaign->scheduled_date = date('Y-m-d');
        $mailTemplateCampaign->scheduled_time = date('H:i');
        $mailTemplateCampaign->total_users = count($usersArray);
        $mailTemplateCampaign->users = $usersArray;
        $mailTemplateCampaign->success_users = [];
        $mailTemplateCampaign->fail_users = [];
        $mailTemplateCampaign->status = MailTemplateCampaignEnum::PENDING;
        $mailTemplateCampaign->schedule_type = 'send_now';
        $mailTemplateCampaign->save();
        $host = request()->getHost();
        (new SendEmailCampaign())->sendEventEmailCampaign($host, $mailTemplateCampaign);
        return;
    }
}
