<?php

namespace App\Http\Controllers\Event;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Event;
use App\Models\EventEPoster;
use App\Models\EventEPosterCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class EventEPosterController extends Controller
{
    use MediaUpload;
    public function index(Request $request)
    {
        $eventEPosters = EventEPoster::where('event_id', Session::get('event_id'));
        $eventEPosters = $eventEPosters->paginate(25);
        return view('event.e-poster.index', compact('request', 'eventEPosters'));
    }

    public function create()
    {
        $eventEPosterCategories = EventEPosterCategory::where('event_id', Session::get('event_id'))
            ->where('status', StatusEnum::ACTIVE)->get();
        return view('event.e-poster.add', compact('eventEPosterCategories'));
    }

    public function store(Request $request, Event $event)
    {
        $request->validate([
            'presenter_id' => ['required'],
            'abstract_id' => ['required'],
            'title' => ['required'],
            'event_e_poster_category_id' => ['required'],
            'institute' => ['required'],
            'index' => ['nullable'],
            'co_author' => ['nullable'],
            'status' => ['required'],
            'poster_file' => ['required', 'max:3072'],
        ]);
        try {
            $data = $request->except(['_token', '_method']);
            if ($request->co_author && is_array($request->co_author) && count($request->co_author) > 0) {
                $data['co_author'] = $request->co_author;
            }
            $data['poster_id'] = $this->upload($request->poster_file)->id;
            $data["index"] = $request->index ?? EventEPoster::max('index') + 1;
            $event->eventEPosters()->create($data);
        } catch (\Throwable $th) {
            return redirect()->route('event.eposter.index')->with('error', $th->getMessage());
        }

        return redirect()->route('event.eposter.index')->with('success', 'E-Poster Created');
    }

    public function edit(EventEPoster $eventEPoster)
    {
        $eventEPosterCategories = EventEPosterCategory::where('event_id', Session::get('event_id'))
            ->where('status', StatusEnum::ACTIVE)->get();
        return view('event.e-poster.edit', compact('eventEPoster', 'eventEPosterCategories'));
    }

    public function update(Request $request, EventEPoster $eventEPoster)
    {
        $request->validate([
            'presenter_id' => ['required'],
            'abstract_id' => ['required'],
            'title' => ['required'],
            'event_e_poster_category_id' => ['required'],
            'institute' => ['required'],
            'index' => ['nullable'],
            'co_author' => ['nullable'],
            'status' => ['required'],
            'poster_file' => ['nullable', 'max:3072'],
        ]);
        try {
            $data = $request->except(['_token', '_method']);
            if ($request->hasFile('poster_file')) {
                $data['poster_id'] = $this->upload($request->poster_file)->id;
            }
            $data["index"] = $request->index ?? EventEPoster::max('index') + 1;
            $eventEPoster->update($data);
        } catch (\Throwable $th) {
            return redirect()->route('event.eposter.index')->with('error', $th->getMessage());
        }

        return redirect()->route('event.eposter.index')->with('success', 'E-Poster Updated');
    }

    public function statusChange(EventEPoster $eventEPoster)
    {
        if ($eventEPoster->status == StatusEnum::ACTIVE) {
            $eventEPoster->status = StatusEnum::INACTIVE;
        } else {
            $eventEPoster->status = StatusEnum::ACTIVE;
        }
        $eventEPoster->save();
        return response()->json(["status" => 200, "message" => "Status Changed"]);
    }

    public function updateIndex(Request $request, EventEPoster $eventEPoster)
    {
        $request->validate([
            'index' => 'required',
        ]);
        try {
            $eventEPoster->index = $request->index;
            $eventEPoster->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('event.eposter.index')->with('success', 'E-Poster Index updated');
        } catch (\Exception $e) {
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }

    public function destroy(EventEPoster $eventEPoster)
    {
        $eventEPoster->delete();
        return response()->json(["status" => 200, "message" => "E-Poster Category Deleted"]);
    }
}
