<?php

namespace App\Http\Controllers\Event;

use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Event;
use App\Models\EventExhibitor;
use App\Models\EventExhibitorCategory;
use App\Models\EventHall;
use App\Models\ScrollerImage;
use App\Rules\DisposableEmailRule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rule;

class EventExhibitorController extends Controller
{
    use MediaUpload;
    public function index(Request $request)
    {
        $eventExhibitors = EventExhibitor::where('event_id', Session::get('event_id'));

        if ($request->filter) {
            $eventExhibitors->where(function ($q) use ($request) {
                $requestContact = preg_replace("/[^0-9]/", "", $request->filter);
                $q->where('name', 'like', "%{$request->filter}%")
                    ->orWhereLike('contact', $requestContact);
            });
        }
        if ($request->email) {
            $eventExhibitors->where(function ($q) use ($request) {
                $q->orWhere('email', $request->email);
            });
        }
        if ($request->status != null) {
            $eventExhibitors->where('status', $request->status);
        }

        $eventExhibitors = $eventExhibitors->paginate(25);
        return view('event.exhibitor.index', compact('request', 'eventExhibitors'));
    }

    public function create(Request $request)
    {
        $eventHalls = EventHall::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        $eventExhibitorCategories = EventExhibitorCategory::where('event_id',Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        return view('event.exhibitor.add', compact('eventHalls','eventExhibitorCategories'));
    }

    public function store(Request $request, Event $event)
    {

        $request->validate([
            'name' => ['required', 'string'],
            'company_name' => ['nullable', 'string'],
            'sponsor_type' => ['nullable', 'string'],
            'event_exhibitor_category_id' => ['required', 'exists:event_exhibitor_categories,id'],
            'email' => ['required', 'unique:event_exhibitors,email', new DisposableEmailRule()],
            'hall_id' => ['nullable', 'exists:event_halls,id'],
            'stall' => ['nullable', 'string'],
            'country_code' => ['required'],
            'contact' => ['required', 'integer'],
            'website_link' => ['nullable', 'url'],
            'brochure_link' => ['nullable', 'url'],
            'about' => ['nullable', 'string'],
            'connect' => ['nullable', 'string'],
            'address' => ['nullable', 'string'],
            'image' => ['required', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);
        if (isset($request->image)) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $data['country_code'] = "+" . str_replace('+', '', $data['country_code']);
        $data["event_id"] = $event->id;
        EventExhibitor::create($data);

        return redirect()->route('event.exhibitors.index')->with('success', 'Exhibitor Created');
    }

    public function edit(EventExhibitor $eventExhibitor)
    {
        $eventHalls = EventHall::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        $eventExhibitorCategories = EventExhibitorCategory::where('event_id',Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        return view('event.exhibitor.edit', compact('eventExhibitor', 'eventHalls','eventExhibitorCategories'));
    }

    public function update(EventExhibitor $eventExhibitor, Request $request)
    {
        $request->validate([
            'name' => ['required', 'string'],
            'company_name' => ['nullable', 'string'],
            'sponsor_type' => ['nullable', 'string'],
            'event_exhibitor_category_id' => ['required', 'exists:event_exhibitor_categories,id'],
            'email' => ['required', 'email', Rule::unique('event_exhibitors')->ignore($eventExhibitor), new DisposableEmailRule()],
            'hall_id' => ['nullable', 'exists:event_halls,id'],
            'stall' => ['nullable', 'string'],
            'country_code' => ['required'],
            'contact' => ['required', 'integer'],
            'website_link' => ['nullable', 'url'],
            'brochure_link' => ['nullable', 'url'],
            'about' => ['nullable', 'string'],
            'connect' => ['nullable', 'string'],
            'address' => ['nullable', 'string'],
            'image' => ['nullable', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);
        if (isset($request->image)) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $data['country_code'] = "+" . str_replace('+', '', $data['country_code']);
        $eventExhibitor->update($data);
        return redirect()->route('event.exhibitors.index')->with('success', 'Exhibitor Updated');
    }

    public function destroy(EventExhibitor $eventExhibitor)
    {
        $eventExhibitor->delete();
        return response()->json(["status" => 200, "message" => "Exhibitor Deleted"]);
    }

    public function imagesIndex(EventExhibitor $eventExhibitor)
    {
        $title = $eventExhibitor->name;
        $addRoute = route('event.exhibitors.images.add', $eventExhibitor->id);
        $scrollImages = $eventExhibitor->scrollerImages()->paginate(25);
        return view('event.scroll-images.index', compact('scrollImages', 'eventExhibitor', 'title', 'addRoute'));
    }

    public function imageAdd(EventExhibitor $eventExhibitor)
    {
        $title = $eventExhibitor->name;
        $storeRoute = route('event.exhibitors.images.store', $eventExhibitor->id);
        return view('event.scroll-images.add', compact('eventExhibitor', 'title', 'storeRoute'));
    }

    public function imagestore(Request $request, EventExhibitor $eventExhibitor)
    {
        $request->validate([
            'image' => ['required', 'image', 'max:2048'],
        ]);
        $scrollerImage = new ScrollerImage([
            'index' => $request->index ?? 0,
            'media_id' => $this->upload($request->image)->id,
        ]);
        $eventExhibitor->scrollerImages()->save($scrollerImage);
        return redirect()->route('event.exhibitors.images.index', $eventExhibitor->id)->with('success', 'Image Created');
    }
}
