<?php

namespace App\Http\Controllers\Event\EventForm;

use App\Enums\EventFormFieldEnum;
use App\Enums\EventFormFieldTypeEnum;
use App\Http\Controllers\Controller;
use App\Models\Event;
use App\Models\EventFormInputField;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rule;

class FormController extends Controller
{
    public function index(Request $request)
    {
        $eventFormStaticFields = EventFormInputField::where('event_id', Session::get('event_id'))->where('field_type', EventFormFieldTypeEnum::STATIC)->get();
        $eventFormDynamicFields = EventFormInputField::where('event_id', Session::get('event_id'))->where('field_type', EventFormFieldTypeEnum::DYNAMIC)->get();
        return view('event.event-form.field.index', compact('eventFormStaticFields', 'eventFormDynamicFields'));
    }

    public function store(Request $request, Event $event)
    {
        $request->validate([
            'display_label' => ['required', 'string'],
            'label' => ['required', 'string'],
            'name' => ['required', 'string'],
            'index' => ['nullable', 'integer'],
            'type' => ['required', Rule::in(['text', 'date', 'textarea', 'radio', 'checkbox', 'dropdown', 'image', 'file_input'])],
            'field_accept_value' => ['nullable', Rule::in([EventFormFieldEnum::FIELD_ACCEPT_ALPHA_NUMBERIC, EventFormFieldEnum::FIELD_ACCEPT_ALPHABETS, EventFormFieldEnum::FIELD_ACCEPT_NUMBERIC])],
        ]);

        $inputField = [];

        $inputField["display_label"] = $request->get('display_label', null);
        $inputField["label"] = $request->get('label', null);
        $inputField["name"] = $request->get('name', null);
        $inputField["default_value"] = $request->get('default_value', null);
        $inputField["helper_text"] = $request->get('helper_text', null);
        $inputField["index"] = $request->index ?? EventFormInputField::max('index') + 1;
        $inputField["type"] = $request->get('type', null);
        $inputField["required"] = $request->get('required', EventFormFieldEnum::NOT_REQUIRED);
        $inputField["visible"] = $request->get('visible', EventFormFieldEnum::NOT_VISIBLE);
        $inputField["field_accept_value"] = $request->get('field_accept_value', null);

        $nonOptionTypes = ['text', 'textarea', 'image', 'date'];
        $inputField["options"] = in_array($request->type, $nonOptionTypes) ? null : $this->getOptionsDetail($request->options);

        if ($request->type == 'image') {
            $inputField["field_accept_value"] = EventFormFieldEnum::FIELD_ACCEPT_IMAGE;
        }
        $event->eventFormInputFields()->create($inputField);

        return redirect()->back()->with('success', 'Field Added');
    }

    public function edit(EventFormInputField $eventFormInputField)
    {
        $eventField = [];
        $eventField['display_label'] = $eventFormInputField->display_label;
        $eventField['label'] = $eventFormInputField->label;
        $eventField['index'] = $eventFormInputField->index;
        $eventField['required'] = $eventFormInputField->required;
        $eventField['visible'] = $eventFormInputField->visible;
        $eventField['helper_text'] = $eventFormInputField->helper_text;
        $eventField['type'] = $eventFormInputField->type;
        $eventField['options'] = $eventFormInputField->options;
        $eventField['name'] = $eventFormInputField->name;
        $eventField['field_accept_value'] = $eventFormInputField->field_accept_value;
        $eventField['default_value'] = $eventFormInputField->default_value;
        $eventField['field_type'] = $eventFormInputField->field_type;

        return response()->json($eventField);
    }

    public function getOptionsDetail(array $data, array $response = [])
    {
        $data = cleanseArray($data);
        if (count($data) > 0) {
            foreach ($data['value'] as $key => $details) {
                $response[$key]['value'] = $details;
                $response[$key]['title'] = $data['title'][$key];
            }
        }
        return $response;
    }

    public function update(EventFormInputField $eventFormInputField, Request $request)
    {
        $request->validate([
            'display_label' => ['required', 'string'],
            'label' => ['required', 'string'],
            'index' => ['nullable', 'integer'],
            'type' => ['required', Rule::in(['text', 'date', 'textarea', 'radio', 'checkbox', 'dropdown', 'image', 'file_input'])],
            'field_accept_value' => ['nullable', Rule::in([EventFormFieldEnum::FIELD_ACCEPT_ALPHA_NUMBERIC, EventFormFieldEnum::FIELD_ACCEPT_ALPHABETS, EventFormFieldEnum::FIELD_ACCEPT_NUMBERIC])],
        ]);
        $inputField = [];
        $inputField["required"] = $request->get('required', EventFormFieldEnum::NOT_REQUIRED);
        $inputField["visible"] = $request->get('visible', EventFormFieldEnum::NOT_VISIBLE);
        $inputField["label"] = $request->get('label', null);
        $inputField["display_label"] = $request->get('display_label', null);
        $inputField["helper_text"] = $request->get('helper_text', null);
        $inputField["index"] = $request->index ?? $eventFormInputField->index;
        $inputField["field_accept_value"] = $request->get('field_accept_value', null);
        $inputField["default_value"] = $request->get('default_value', null);

        $nonOptionTypes = ['text', 'textarea', 'image', 'date'];
        $inputField["options"] = in_array($request->type, $nonOptionTypes) ? null : $this->getOptionsDetail($request->options ?? []);

        if ($request->type == 'image') {
            $inputField["field_accept_value"] = EventFormFieldEnum::FIELD_ACCEPT_IMAGE;
        }

        $eventFormInputField->update($inputField);
        if ($request->ajax()) {
            return response()->json("success");
        } else {
            return redirect()->back()->with('success', 'Fields Updated');
        }
    }

    public function destroy(EventFormInputField $eventFormInputField)
    {
        $eventFormInputField->delete();
        return response()->json("success");
    }

    public function addStaticField()
    {
        $eventId = Session::get('event_id');

        $fields = [
            [
                'display_label' => 'Salutation',
                'label' => 'Salutation',
                'name' => 'salutation',
                'type' => 'dropdown',
                'options' => [
                    ['value' => 'Dr.', 'title' => 'Dr.'],
                    ['value' => 'Mr.', 'title' => 'Mr.'],
                    ['value' => 'Mrs.', 'title' => 'Mrs.'],
                    ['value' => 'Ms.', 'title' => 'Ms.'],
                    ['value' => 'Prof.', 'title' => 'Prof.'],
                ],
            ],
            [
                'display_label' => 'Full Name(as required on the certificate)',
                'label' => 'Full Name(as required on the certificate)',
                'name' => 'full_name',
                'type' => 'text',
                'helper_text' => null,
                'options' => null,
            ],
            [
                'display_label' => 'Email Id',
                'label' => 'Email Id',
                'name' => 'email',
                'type' => 'text',
                'helper_text' => null,
                'options' => null,
            ],
            [
                'display_label' => 'Country',
                'label' => 'Country',
                'name' => 'country',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => null,
            ],
            [
                'display_label' => 'Mobile No.(whatsapp Number only without country code)',
                'label' => 'Mobile No.(whatsapp Number only without country code)',
                'name' => 'mobile',
                'type' => 'text',
                'helper_text' => null,
                'options' => null,
            ],
            [
                'display_label' => 'Gender',
                'label' => 'Gender',
                'name' => 'gender',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => [
                    ['value' => 'Male', 'title' => 'Male'],
                    ['value' => 'Female', 'title' => 'Female'],
                ],
            ],
            [
                'display_label' => 'Meal preference',
                'label' => 'Meal preference',
                'name' => 'meal_preference',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => [
                    ['value' => 'Veg', 'title' => 'Veg'],
                    ['value' => 'Non Veg', 'title' => 'Non Veg'],
                ],
            ],
            [
                'display_label' => 'Institute',
                'label' => 'Institute',
                'name' => 'institute',
                'type' => 'text',
                'helper_text' => null,
                'options' => null,
            ],
            [
                'display_label' => 'Address',
                'label' => 'Address',
                'name' => 'address',
                'type' => 'text',
                'helper_text' => null,
                'options' => null,
            ],
            [
                'display_label' => 'City',
                'label' => 'City',
                'name' => 'city',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => null,
            ],
            [
                'display_label' => 'State',
                'label' => 'State',
                'name' => 'state',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => null,
            ],
            [
                'display_label' => 'Medical Council Registration Number',
                'label' => 'Medical Council Registration Number',
                'name' => 'medical_council_registration_number',
                'type' => 'text',
                'helper_text' => null,
                'options' => null,
            ],
            [
                'display_label' => 'Category',
                'label' => 'Category',
                'name' => 'category',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => null,
            ],
            [
                'display_label' => 'Membership No.',
                'label' => 'Membership No.',
                'name' => 'membership_no',
                'type' => 'text',
                'helper_text' => null,
                'options' => null,
            ],
            [
                'display_label' => 'Do you want to register Accompany?',
                'label' => 'Do you want to register Accompany?',
                'name' => 'register_accompany',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => [
                    ['value' => 'Yes', 'title' => 'Yes'],
                    ['value' => 'No', 'title' => 'No'],
                ],
            ],
            [
                'display_label' => 'No of Accompanying Persons?',
                'label' => 'No of Accompanying Persons?',
                'name' => 'accompany_person',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => [
                    ['value' => '1', 'title' => '1'],
                    ['value' => '2', 'title' => '2'],
                    ['value' => '3', 'title' => '3'],
                ],
            ],
            [
                'display_label' => 'Do you want attend Workshop?',
                'label' => 'Do you want attend Workshop?',
                'name' => 'workshop',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => null,
            ],
            [
                'display_label' => 'Payment Mode',
                'label' => 'Payment Mode',
                'name' => 'payment_mode',
                'type' => 'dropdown',
                'helper_text' => '--Select--',
                'options' => [
                    ['value' => 'Offline Registration', 'title' => 'Offline Registration'],
                    ['value' => 'Online Registration', 'title' => 'Online Registration'],
                ],
            ],
            [
                'display_label' => 'UTR Id / Transaction Id.',
                'label' => 'UTR Id / Transaction Id.',
                'name' => 'transaction_id',
                'type' => 'text',
                'helper_text' => null,
                'options' => null,
            ],
            [
                'display_label' => 'Transaction Date',
                'label' => 'Transaction Date',
                'name' => 'transaction_date',
                'type' => 'date',
                'helper_text' => 'dd-mm-yyyy',
                'options' => null,
            ],
            [
                'display_label' => 'Upload Payment Receipt',
                'label' => 'Upload Payment Receipt',
                'name' => 'upload_payment_receipt',
                'type' => 'file_input',
                'helper_text' => null,
                'options' => null,
            ],
        ];

        foreach ($fields as $index => $field) {
            EventFormInputField::updateOrCreate(
                [
                    'event_id' => $eventId,
                    'field_type' => EventFormFieldTypeEnum::STATIC,
                    'label' => $field['label'],
                ],
                [
                    'display_label' => $field['display_label'],
                    'name' => $field['name'],
                    'type' => $field['type'],
                    'required' => EventFormFieldEnum::REQUIRED,
                    'visible' => EventFormFieldEnum::VISIBLE,
                    'default_value' => null,
                    'helper_text' => $field['helper_text'] ?? null,
                    'index' => $index + 1,
                    'field_accept_value' => EventFormFieldEnum::FIELD_ACCEPT_ALPHA_NUMBERIC,
                    'options' => $field['options'],
                    'field_type' => EventFormFieldTypeEnum::STATIC,
                    'event_id' => $eventId,
                ]
            );
        }

        return response()->json(['message' => 'Static fields added successfully']);
    }


    public function addStaticFieldOld()
    {
        $data = [
            'event_id' => Session::get('event_id'),
            'label' => 'Salutation',
            'name' => 'salutation',
            'type' => 'dropdown',
            'required' => EventFormFieldEnum::NOT_REQUIRED,
            'visible' => EventFormFieldEnum::VISIBLE,
            'default_value' => '',
            'helper_text' => null,
            'index' => 1,
            'field_accept_value' => null,
            'options' => [
                [
                    "value" => "Dr.",
                    "title" => "Dr."
                ],
                [
                    "value" => "Mr.",
                    "title" => "Mr."
                ],
                [
                    "value" => "Mrs.",
                    "title" => "Mrs."
                ],
                [
                    "value" => "Ms.",
                    "title" => "Ms."
                ],
                [
                    "value" => "Prof.",
                    "title" => "Prof."
                ],
            ],
            'field_type' => EventFormFieldTypeEnum::STATIC,
        ];
        return EventFormInputField::create($data);
    }
}
