<?php

namespace App\Http\Controllers\Event\EventForm;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Models\Event;
use App\Models\EventFormCategory;
use App\Models\EventFormSlab;
use App\Models\EventFormSlabCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class SlabController extends Controller
{
    public function index(Request $request)
    {
        $eventFormSlabs = EventFormSlab::where('event_id', Session::get('event_id'));
        $eventFormSlabs = $eventFormSlabs->paginate(25);
        return view('event.event-form.slab.index', compact('request', 'eventFormSlabs'));
    }

    public function create()
    {
        $eventFormCategories = EventFormCategory::where('event_id', Session::get('event_id'))->get();
        return view('event.event-form.slab.create', compact('eventFormCategories'));
    }

    public function store(Request $request, Event $event)
    {
        $data = $request->validate([
            'name' => 'required|string',
            'start_date' => 'required',
            'start_time' => 'required|date_format:H:i',
            'end_date' => 'required',
            'end_time' => 'required|date_format:H:i',
        ]);
        $data['start_date'] = date('Y-m-d', strtotime($request->start_date));
        $data['end_date'] = date('Y-m-d', strtotime($request->end_date));
        $slab = $event->eventFormSlabs()->create($data);

        $allCategories = $request->input('categories', []);
        $checkedCategories = array_filter($allCategories, function ($category) {
            return isset($category['checked']) && $category['checked'] == 1;
        });
    
        foreach ($checkedCategories as $category) {
            EventFormSlabCategory::create([
                'event_form_slab_id' => $slab->id,
                'event_form_category_id' => $category['id'],
                'event_id' => $event->id,
                'amount' => $category['amount'],
            ]);
        }

        return redirect()->route('event.forms.slabs.index')->with('success', 'Slab Created');
    }

    public function edit(EventFormSlab $eventFormSlab)
    {
        $eventId = Session::get('event_id');
        $eventFormCategories = EventFormCategory::where('event_id', $eventId)->get();

        $selectedCategories = EventFormSlabCategory::where('event_form_slab_id', $eventFormSlab->id)
            ->pluck('amount', 'event_form_category_id')
            ->toArray();
        return view('event.event-form.slab.edit', compact('eventFormSlab', 'eventFormCategories', 'selectedCategories'));
    }
    public function update(EventFormSlab $eventFormSlab, Request $request)
    {
        DB::beginTransaction();

        try {
            $data = $request->validate([
                'name' => 'required|string',
                'start_date' => 'required',
                'start_time' => 'required|date_format:H:i',
                'end_date' => 'required',
                'end_time' => 'required|date_format:H:i',
            ]);
            $data['start_date'] = date('Y-m-d', strtotime($request->start_date));
            $data['end_date'] = date('Y-m-d', strtotime($request->end_date));
            $eventFormSlab->update($data);

            $eventId = Session::get('event_id');
            $existingCategories = EventFormSlabCategory::where('event_form_slab_id', $eventFormSlab->id)
            ->get()
            ->keyBy('event_form_category_id');

            $submittedCategories = $request->input('categories', []);

            foreach ($submittedCategories as $cat) {
                $categoryId = $cat['id'];
                $isChecked = isset($cat['checked']) && $cat['checked'] == 1;
                $amount = $cat['amount'] ?? null;
        
                // A. If checked and already exists → update amount if changed
                if ($isChecked && isset($existingCategories[$categoryId])) {
                    $existing = $existingCategories[$categoryId];
                    if ($existing->amount != $amount) {
                        $existing->update(['amount' => $amount]);
                    }
                    // Remove from list to track unhandled (unchecked) ones
                    unset($existingCategories[$categoryId]);
                }
        
                // B. If checked and doesn't exist → insert new
                elseif ($isChecked && !isset($existingCategories[$categoryId])) {
                    EventFormSlabCategory::create([
                        'event_form_slab_id' => $eventFormSlab->id,
                        'event_form_category_id' => $categoryId,
                        'event_id' => $eventId,
                        'amount' => $amount,
                    ]);
                }
            }
            foreach ($existingCategories as $leftover) {
                $leftover->delete();
            }

            DB::commit();

            return redirect()->route('event.forms.slabs.index')->with('success', 'Slab Updated');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'An error occurred while updating the slab. Please try again.']);
        }
    }
    public function statusChange(EventFormSlab $eventFormSlab)
    {
        if ($eventFormSlab->status == StatusEnum::ACTIVE) {
            $eventFormSlab->status = StatusEnum::INACTIVE;
        } else {
            $eventFormSlab->status = StatusEnum::ACTIVE;
        }
        $eventFormSlab->save();
        return response()->json(["status" => 200, "message" => "Status Changed"]);
    }
    public function destroy(EventFormSlab $eventFormSlab)
    {
        $eventFormSlab->delete();
        return response()->json(["status" => 200, "message" => "Slab Deleted"]);
    }
}
