<?php

namespace App\Http\Controllers\Event;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\Event;
use App\Models\EventMailTemplate;
use Illuminate\Http\Request;
use App\Rules\DisposableEmailRule;
use Illuminate\Support\Facades\Session;

class EventMailTemplateController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $mailTemplates = EventMailTemplate::where('event_id', Session::get('event_id'))
            ->latest();

        if ($request->name) {
            $mailTemplates->where("name", 'like', '%' . $request->name . '%');
        }
        if ($request->template_id) {
            $mailTemplates->where("template_id", 'like', '%' . $request->template_id . '%');
        }
        if ($request->subject) {
            $mailTemplates->where("subject", 'like', '%' . $request->subject . '%');
        }

        $mailTemplates = $mailTemplates->paginate(25);

        return view('event.mail-templates.index', compact('mailTemplates', 'request'));
    }

    public function create()
    {
        $columnsDisplay = [
            'name',
            'contact',
        ];
        $mailjetTemplates = getMailjetTemplates();
        return view('event.mail-templates.create', compact('columnsDisplay', 'mailjetTemplates'));
    }

    public function store(Request $request, Event $event)
    {
        $data = $request->validate([
            'name' => ['required'],
            'template_id' => ['required'],
            'from' => ['required'],
            'subject' => ['required'],
            'cc' => ['required', 'array'],
            'cc.*' => ['email', new DisposableEmailRule()],
            'bcc' => ['required', 'array'],
            'bcc.*' => ['email', new DisposableEmailRule()],
            'dynamic_variables' => ['nullable', 'array'],
            'status' => ['required'],
            'attachments' => ['nullable', 'array'],
            'attachments.*' => ['file'],
        ], [
            'cc.*' => 'Disposable emails are not allowed in CC.',
            'bcc.*' => 'Disposable emails are not allowed in BCC.',
        ]);

        $attachmentArray = [];
        if ($request->attachments && $request->attachments > 0) {
            foreach ($request->attachments as $attachment) {
                $size = $attachment->getSize() / 1024 / 1024;
                if ($size <= 2) {
                    $media = $this->upload($attachment)->id;
                    $attachmentArray[] = $media;
                }
            }
            $data['attachment'] = $attachmentArray;
        }
        $event->eventMailTemplates()->create($data);
        return redirect()->route('event.mail.templates.index')->with('success', 'Email Template Created');
    }

    public function edit(EventMailTemplate $mailTemplate)
    {
        $columnsDisplay = [
            'name',
            'contact',
        ];
        $mailjetTemplates = getMailjetTemplates();
        return view('event.mail-templates.edit', compact('mailTemplate', 'columnsDisplay', 'mailjetTemplates'));
    }

    public function update(EventMailTemplate $mailTemplate, Request $request)
    {
        $data = $request->validate([
            'name' => ['required'],
            'template_id' => ['required'],
            'from' => ['required'],
            'subject' => ['required'],
            'cc' => ['required', 'array'],
            'cc.*' => ['email', new DisposableEmailRule()],
            'bcc' => ['required', 'array'],
            'bcc.*' => ['email', new DisposableEmailRule()],
            'dynamic_variables' => ['nullable', 'array'],
            'status' => ['required'],
            'attachments' => ['nullable', 'array'],
            'attachments.*' => ['file'],
        ], [
            'cc.*' => 'Disposable emails are not allowed in CC.',
            'bcc.*' => 'Disposable emails are not allowed in BCC.',
        ]);

        $attachmentArray = [];
        if ($request->attachments && $request->attachments > 0) {
            foreach ($request->attachments as $attachment) {
                $size = $attachment->getSize() / 1024 / 1024;
                if ($size <= 2) {
                    $media = $this->upload($attachment)->id;
                    $attachmentArray[] = $media;
                }
            }
            $data['attachment'] = $attachmentArray;
            if ($mailTemplate->getAttachments()) {
                foreach ($mailTemplate->getAttachments() as $attachment) {
                    if ($attachment && $attachment->path) {
                        $this->remove($attachment->path);
                    }
                }
            }
        }
        $mailTemplate->update($data);
        return redirect()->route('event.mail.templates.index')->with('success', 'Email Template Updated');
    }

    public function statusChange(EventMailTemplate $mailTemplate)
    {
        if ($mailTemplate->status == StatusEnum::ACTIVE) {
            $mailTemplate->status = StatusEnum::INACTIVE;
        } else {
            $mailTemplate->status = StatusEnum::ACTIVE;
        }
        $mailTemplate->save();
        return response()->json(["status" => 200, "message" => "Status Changed"]);
    }

    public function destroy(EventMailTemplate $mailTemplate)
    {
        if ($mailTemplate->getAttachments()) {
            foreach ($mailTemplate->getAttachments() as $attachment) {
                if ($attachment && $attachment->path) {
                    $this->remove($attachment->path);
                    $attachment->delete();
                }
            }
        }
        $mailTemplate->delete();
        return response()->json(["status" => 200, "message" => "Mail Template Deleted"]);
    }
}
