<?php

namespace App\Http\Controllers\Event;

use App\Enums\EventAskQuestionEnum;
use App\Enums\EventSessionEnum;
use App\Enums\RoleEnum;
use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Models\Event;
use App\Models\EventDay;
use App\Models\EventDesignation;
use App\Models\EventFaculty;
use App\Models\EventHall;
use App\Models\EventSession;
use App\Models\EventSessionUser;
use App\Models\EventSpeaker;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rule;

class EventSessionController extends Controller
{
    public function index(Request $request)
    {
        $eventSessions = EventSession::where('event_id', Session::get('event_id'))->whereNull('parent_id');
        if ($request->title) {
            $eventSessions->where('title', 'like', "%{$request->title}%");
        }
        if ($request->event_hall_id) {
            $eventSessions->whereHas('eventHall', function ($q) use ($request) {
                $q->where('id', $request->event_hall_id);
            });
        }
        if ($request->event_day_id) {
            $eventSessions->whereHas('eventDay', function ($q) use ($request) {
                $q->where('id', $request->event_day_id);
            });
        }
        $eventSessions = $eventSessions->paginate(25);
        $eventHalls = EventHall::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        $eventDays = EventDay::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        return view('event.session.index', compact('request', 'eventSessions', 'eventHalls', 'eventDays'));
    }

    public function create(Request $request)
    {
        $eventSpeakers = EventSpeaker::where('event_id', Session::get('event_id'))
            ->select('id', 'title', 'name')
            ->get();
        $eventDays = EventDay::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        $eventHalls = EventHall::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        $eventDesignations = EventDesignation::where('event_id', Session::get('event_id'))
            ->select('id', 'designation')
            ->get();

        return view('event.session.add', compact('eventSpeakers', 'eventDays', 'eventHalls', 'eventDesignations'));
    }

    public function store(Request $request, Event $event)
    {
        $data = $request->validate([
            'title' => ['required', 'string'],
            'topic' => ['nullable'],
            'notes' => ['nullable'],
            'type' => ['required', Rule::in([EventSession::TYPE_AUTO, EventSession::TYPE_MANUAL])],
            'start_time' => ['required', 'date_format:H:i'],
            'end_time' => ['required', 'date_format:H:i'],
            'sub_sessions_min' => ['required_if:type,' . EventSession::TYPE_AUTO],
            'sub_sessions' => ['required_if:type,' . EventSession::TYPE_AUTO],
            'event_day_id' => ['nullable', 'exists:event_days,id'],
            'event_hall_id' => ['nullable', 'exists:event_halls,id'],
            'event_speaker_ids.*' => ['nullable', 'exists:event_speakers,id'],
            'text_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'bg_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'ask_questions' => ['required'],
            'moderator' => ['nullable', 'string'],
        ]);
        $data['ask_questions'] = $request->ask_questions == StatusEnum::DEACTIVE ? EventAskQuestionEnum::NO : EventAskQuestionEnum::YES;
        $data['event_id'] = $event->id;
        $data['start_time'] = Carbon::createFromFormat('H:i', $request->start_time);
        $data['end_time'] = Carbon::createFromFormat('H:i', $request->end_time);
        // create Main/Parent Event Session
        $eventSession = $event->eventSessions()->create($data);

        if ($data['type'] == EventSession::TYPE_AUTO) {
            // create Sub Event Session
            $subSessions = $this->makeSubsessions($eventSession->id, $data['start_time'], $data['end_time'], $data, true); //$data

            // create Sub Event Session
            $event->eventSessions()->insert($subSessions);
        }

        $this->setSessionSpeakers($eventSession, $request);

        // create Session Users in => event_session_users
        $this->setSessionUsers($eventSession, $request);

        return redirect()->route('event.sessions.index')->with('success', 'Session Created');
    }


    private function makeSubsessions($parentId, Carbon $strTime, Carbon $endTime, array $data, bool $includeTitle = true)
    {
        $subSessionMin = (int) $data['sub_sessions_min'];
        $totalMinutes = $strTime->diffInMinutes($endTime);  // Difference in minutes
        $totalSessions = intdiv($totalMinutes, $subSessionMin); // Calculate how many sub-sessions are possible
        $sessionStartTimes = range(0, $totalSessions - 1);
        return collect($sessionStartTimes)->map(function ($i) use ($strTime, $subSessionMin, $parentId, $data, $includeTitle) {
            $sessionStartTime = $strTime->copy()->addMinutes($i * $subSessionMin);
            $sessionEndTime = $sessionStartTime->copy()->addMinutes($subSessionMin);

            $session = [
                'parent_id' => $parentId,
                'event_id' => $data['event_id'],
                'start_time' => $sessionStartTime->format('H:i'),
                'end_time' => $sessionEndTime->format('H:i'),
                'event_day_id' => $data['event_day_id'] ?? null,
                'event_hall_id' => $data['event_hall_id'] ?? null,
                'text_color' => $data['text_color'] ?? null,
                'bg_color' => $data['bg_color'] ?? null,
                'ask_questions' => EventAskQuestionEnum::NO,
                'created_at' => now(),
                'updated_at' => now(),
            ];

            if ($includeTitle) {
                $session['title'] = $data['title'] ?? null;
            }

            return $session;
        })->toArray();
    }

    public function edit(EventSession $eventSession)
    {
        $eventSpeakers = EventSpeaker::where('event_id', Session::get('event_id'))
            ->select('id', 'title', 'name')
            ->get();
        $eventDays = EventDay::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        $eventHalls = EventHall::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        $eventDesignations = EventDesignation::where('event_id', Session::get('event_id'))
            ->select('id', 'designation')
            ->get();

        return view('event.session.edit', compact('eventSession', 'eventSpeakers', 'eventDays', 'eventHalls', 'eventDesignations'));
    }

    // public function update(Request $request, EventSession $eventSession)
    // {
    //     $data = $request->validate([
    //         'title' => ['required', 'string'],
    //         'topic' => ['nullable'],
    //         'notes' => ['nullable'],
    //         'type' => ['required', Rule::in([EventSession::TYPE_AUTO, EventSession::TYPE_MANUAL])],
    //         'start_time' => ['required', 'date_format:H:i'],
    //         'end_time' => ['required', 'date_format:H:i'],
    //         'sub_sessions_min' => ['required_if:type,' . EventSession::TYPE_AUTO],
    //         'sub_sessions' => ['required_if:type,' . EventSession::TYPE_AUTO],
    //         'event_day_id' => ['nullable', 'exists:event_days,id'],
    //         'event_hall_id' => ['nullable', 'exists:event_halls,id'],
    //         'event_faculty_ids.*' => ['nullable', 'exists:event_faculties,id'],
    //         'text_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
    //         'bg_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
    //         'ask_questions' => ['required'],
    //     ]);

    //     // Update main event session data
    //     $data['start_time'] = Carbon::createFromFormat('H:i', $request->start_time);
    //     $data['end_time'] = Carbon::createFromFormat('H:i', $request->end_time);
    //     $data['event_id'] = $eventSession->event_id;
    //     if($eventSession->start_time != $data['start_time']->format('H:i') || $eventSession->end_time != $data['end_time']->format('H:i')){
    //         dd("!");
    //     } else {
    //         dd("2");
    //     }
    //     // Update the main session
    //     $eventSession->update($data);

    //     if ($data['type'] == EventSession::TYPE_AUTO) {
    //         $eventSession->subSessions()->delete();
    //         // Recreate Sub Event Sessions
    //         $subSessions = $this->makeSubsessions($eventSession->id, $data['start_time'], $data['end_time'], $data);
    //         $eventSession->subSessions()->insert($subSessions->toArray());
    //     } else if ($data['type'] == EventSession::TYPE_MANUAL) {
    //         $eventSession->subSessions()->delete();
    //     }

    //     $this->setSessionFaculties($eventSession, $request);

    //     $this->setSessionUsers($eventSession, $request);
    //     return redirect()->route('event.sessions.index')->with('success', 'Session Updated');
    // }

    public function update(Request $request, EventSession $eventSession)
    {
        $data = $request->validate([
            'title' => ['required', 'string'],
            'topic' => ['nullable'],
            'notes' => ['nullable'],
            'type' => ['required', Rule::in([EventSession::TYPE_AUTO, EventSession::TYPE_MANUAL])],
            'start_time' => ['required', 'date_format:H:i'],
            'end_time' => ['required', 'date_format:H:i'],
            'sub_sessions_min' => ['required_if:type,' . EventSession::TYPE_AUTO],
            'sub_sessions' => ['required_if:type,' . EventSession::TYPE_AUTO],
            'event_day_id' => ['nullable', 'exists:event_days,id'],
            'event_hall_id' => ['nullable', 'exists:event_halls,id'],
            'event_speaker_ids.*' => ['nullable', 'exists:event_speakers,id'],
            'text_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'bg_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'ask_questions' => ['required'],
            'moderator' => ['nullable', 'string'],
        ]);

        // Update main event session data
        $data['ask_questions'] = $request->ask_questions == StatusEnum::DEACTIVE ? EventAskQuestionEnum::NO : EventAskQuestionEnum::YES;
        $data['start_time'] = Carbon::createFromFormat('H:i', $request->start_time);
        $data['end_time'] = Carbon::createFromFormat('H:i', $request->end_time);
        $data['event_id'] = $eventSession->event_id;

        // Update the main session
        $eventSession->update($data);

        if ($data['type'] == EventSession::TYPE_AUTO) {
            // Check existing sub-sessions
            $existingSubSessions = $eventSession->subSessions()->get();
            $newSubSessions = $this->makeSubsessions($eventSession->id, $data['start_time'], $data['end_time'], $data, false);

            // Update existing sub-sessions if necessary
            foreach ($existingSubSessions as $index => $subSession) {
                if (isset($newSubSessions[$index])) {
                    // Update existing sub-session
                    $subSession->update($newSubSessions[$index]);
                } else {
                    $subSession->delete();
                }
            }

            // Add new sub-sessions if there are more than existing
            if (count($newSubSessions) > count($existingSubSessions)) {
                $additionalSubSessions = array_slice($newSubSessions, count($existingSubSessions));
                $eventSession->subSessions()->insert($additionalSubSessions);
            }
        } else if ($data['type'] == EventSession::TYPE_MANUAL) {
            $eventSession->subSessions()->delete();
        }

        $this->setSessionSpeakers($eventSession, $request);
        $this->setSessionUsers($eventSession, $request);
        return redirect()->route('event.sessions.index')->with('success', 'Session Updated');
    }

    public function destroy(EventSession $eventSession)
    {
        $eventSession->delete();
        return response()->json(["status" => 200, "message" => "Session Deleted"]);
    }

    public function statusChange(Request $request, EventSession $eventSession)
    {
        $request->validate([
            'status' => ['required', Rule::in([EventSessionEnum::UPCOMING_STATUS, EventSessionEnum::ONGOING_STATUS, EventSessionEnum::COMPLETED_STATUS])],
        ]);
        $eventSession->update(['status' => $request->status]);
        if ($request->ajax()) {
            return response()->json(['message' => 'Session status changed successfully']);
        } else {
            return back()->with('success', 'Session status changed successfully');
        }
    }

    ///*****SUB_SESSIONS******///

    public function subSesssionIndex(Request $request, EventSession $eventSession)
    {
        $parentSession = $eventSession;
        $eventSessions = $eventSession->subSessions();
        if ($request->title) {
            $eventSessions->where('title', 'like', "%{$request->title}%");
        }
        if ($request->event_hall_id) {
            $eventSessions->whereHas('eventHall', function ($q) use ($request) {
                $q->where('id', $request->event_hall_id);
            });
        }
        if ($request->event_day_id) {
            $eventSessions->whereHas('eventDay', function ($q) use ($request) {
                $q->where('id', $request->event_day_id);
            });
        }
        $eventSessions = $eventSessions->paginate(25);
        $eventHalls = EventHall::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        $eventDays = EventDay::where('event_id', Session::get('event_id'))
            ->select('id', 'name')
            ->get();
        return view('event.session.index', compact('request', 'eventSessions', 'parentSession', 'eventHalls', 'eventDays'));
    }

    public function eventSessionCreate(Request $request, EventSession $eventSession)
    {
        $parentSession = $eventSession;
        $eventSpeakers = EventSpeaker::where('event_id', Session::get('event_id'))
            ->select('id', 'title', 'name')
            ->get();
        $eventDesignations = EventDesignation::where('event_id', Session::get('event_id'))
            ->select('id', 'designation')
            ->get();

        return view('event.session.sub-session-add', compact('parentSession', 'eventSpeakers', 'eventDesignations'));
    }

    public function subSessionEdit(EventSession $eventSession)
    {
        $eventSpeakers = EventSpeaker::where('event_id', Session::get('event_id'))
            ->select('id', 'title', 'name')
            ->get();

        $eventDesignations = EventDesignation::where('event_id', Session::get('event_id'))
            ->select('id', 'designation')
            ->get();
        return view('event.session.sub-session-edit', compact('eventSession', 'eventSpeakers', 'eventDesignations'));
    }

    public function subSessionStore(Request $request, EventSession $eventSession)
    {
        $parentSession = $eventSession;

        $data = $request->validate([
            'title' => ['required', 'string'],
            'topic' => ['nullable'],
            'notes' => ['nullable'],
            'start_time' => ['required', 'date_format:H:i'],
            'end_time' => ['required', 'date_format:H:i'],
            'event_speaker_ids.*' => ['nullable', 'exists:event_speakers,id'],
            'text_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'bg_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'ask_questions' => ['required'],
            'moderator' => ['nullable', 'string'],
        ]);
        $data['ask_questions'] = $request->ask_questions == StatusEnum::DEACTIVE ? EventAskQuestionEnum::NO : EventAskQuestionEnum::YES;
        $data['start_time'] = Carbon::createFromFormat('H:i', $request->start_time);
        $data['end_time'] = Carbon::createFromFormat('H:i', $request->end_time);

        $data['event_id'] = $parentSession->event_id ?? null;
        $data['event_hall_id'] = $parentSession->event_hall_id ?? null;
        $data['event_day_id'] = $parentSession->event_day_id ?? null;

        $eventSession = $parentSession->subSessions()->create($data);

        // $this->setSessionFaculties($eventSession, $request);
        $this->setSessionSpeakers($eventSession, $request);
        // create Session Users in => event_session_users

        $this->setSessionUsers($eventSession, $request);
        return redirect()->route('event.sub.sessions.index', $parentSession->id)->with('success', 'Session Created');
    }

    public function subSessionUpdate(Request $request, EventSession $eventSession)
    {
        $data = $request->validate([
            'title' => ['required', 'string'],
            'topic' => ['nullable'],
            'notes' => ['nullable'],
            'start_time' => ['required', 'date_format:H:i'],
            'end_time' => ['required', 'date_format:H:i'],
            'event_speaker_ids.*' => ['nullable', 'exists:event_speakers,id'],
            'text_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'bg_color' => ['nullable', 'regex:/^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'ask_questions' => ['required'],
            'moderator' => ['nullable', 'string'],
        ]);
        $data['ask_questions'] = $request->ask_questions == StatusEnum::DEACTIVE ? EventAskQuestionEnum::NO : EventAskQuestionEnum::YES;
        $data['start_time'] = Carbon::createFromFormat('H:i', $request->start_time);
        $data['end_time'] = Carbon::createFromFormat('H:i', $request->end_time);

        // create Main/Parent Event Session
        $eventSession->update($data);

        // $this->setSessionFaculties($eventSession, $request);
        $this->setSessionSpeakers($eventSession, $request);
        // create Session Users in => event_session_users
        $this->setSessionUsers($eventSession, $request);
        return redirect()->route('event.sub.sessions.index', $eventSession->parent_id)->with('success', 'Session Created');
    }

    protected function setSessionFaculties(EventSession $eventSession, Request $request)
    {
        if ($request->event_faculty_ids) {
            // create Session Faculty in => event_session_users
            $eventSession->eventSessionFaculties()->sync($request->event_faculty_ids);
        } else {
            $eventSession->eventSessionFaculties()->detach();
        }
    }
    protected function setSessionSpeakers(EventSession $eventSession, Request $request)
    {
        if ($request->event_speaker_ids) {
            // create Session Speaker in => event_session_users
            $eventSession->eventSessionSpeakers()->sync($request->event_speaker_ids);
        } else {
            $eventSession->eventSessionSpeakers()->detach();
        }
    }
    protected function setSessionUsers(EventSession $eventSession, Request $request)
    {
        $eventDesignations = EventDesignation::where('event_id', Session::get('event_id'))
            ->select('id', 'designation')
            ->get();

        foreach ($eventDesignations as $eventDesignation) {
            if ($request->event_designations) {
                $eventDesignation->eventSessionUsers()->where('event_session_id', $eventSession->id)->delete();
                if (isset($request->event_designations[$eventDesignation->id]) && is_array($request->event_designations[$eventDesignation->id])) {
                    foreach ($request->event_designations[$eventDesignation->id] as $title)
                        if ($title) {
                            EventSessionUser::create([
                                'event_session_id' => $eventSession->id,
                                'event_designation_id' => $eventDesignation->id,
                                'title' => $title,
                            ]);
                        }
                }
            } else {
                $eventDesignation->eventSessionUsers()->where('event_session_id', $eventSession->id)->delete();
            }
        }
    }

    public function checkSessionUser(Request $request, Event $event, EventSession $eventSession = null)
    {
        $request->validate([
            'start_time' => ['required', 'date_format:H:i'],
            'end_time' => ['required', 'date_format:H:i'],
            'user_id' => ['required', 'exists:users,id'],
            'event_day_id' => ['required', 'exists:event_days,id'],
        ]);

        $startTime = Carbon::createFromFormat('H:i', $request->start_time);
        $endTime = Carbon::createFromFormat('H:i', $request->end_time);

        $eventSessionExists = EventSession::where('event_id', $event->id)
            ->where(function ($query) use ($startTime, $endTime) {
                $query->whereBetween('start_time', [$startTime, $endTime])
                    ->orWhereBetween('end_time', [$startTime, $endTime]);
            })->whereHas('eventDay', function ($query) use ($request) {
                $query->where('event_day_id', $request->event_day_id);
            })->whereHas('eventSessionUsers', function ($query) use ($request) {
                $query->where('user_id', $request->user_id);
            })
            ->when($eventSession, function ($query) use ($eventSession) {
                $query->where('id', '!=', $eventSession->id);
            })
            ->exists();

        if ($eventSessionExists) {
            return response()->json([
                'error' => true,
                'message' => "This Member is Already Assigned for this time slot",
            ]);
        } else {
            return response()->json([
                'error' => false,
                'message' => "This Member is Available for this time slot",
            ]);
        }
    }

    public function checkSpeakerUser(Request $request, Event $event, EventSession $eventSession = null)
    {
        $request->validate([
            'start_time' => ['required', 'date_format:H:i'],
            'end_time' => ['required', 'date_format:H:i'],
            'speaker_id' => ['required', 'exists:event_speakers,id'],
            'event_day_id' => ['required', 'exists:event_days,id'],
        ]);

        $startTime = Carbon::createFromFormat('H:i', $request->start_time);
        $endTime = Carbon::createFromFormat('H:i', $request->end_time);

        $eventSessionSpeakerExists = EventSession::where('event_id', $event->id)
            ->where(function ($query) use ($startTime, $endTime) {
                $query->whereBetween('start_time', [$startTime, $endTime])
                    ->orWhereBetween('end_time', [$startTime, $endTime]);
            })->whereHas('eventDay', function ($query) use ($request) {
                $query->where('event_day_id', $request->event_day_id);
            })->whereHas('eventSessionSpeakers', function ($query) use ($request) {
                $query->where('event_speaker_id', $request->speaker_id);
            })
            ->when($eventSession, function ($query) use ($eventSession) {
                $query->where('id', '!=', $eventSession->id);
            })
            ->exists();

        if ($eventSessionSpeakerExists) {
            return response()->json([
                'error' => true,
                'message' => "This Speaker is Already Assigned for this time slot",
            ]);
        } else {
            return response()->json([
                'error' => false,
                'message' => "This Speaker is Available for this time slot",
            ]);
        }
    }
}
