<?php

namespace App\Http\Controllers\Event;

use App\Enums\StatusEnum;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Imports\EventSpeakerImport;
use App\Models\Event;
use App\Models\EventSpeaker;
use App\Models\EventSpeakerType;
use App\Rules\DisposableEmailRule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;

class SpeakerController extends Controller
{
    use MediaUpload;

    public function index(Request $request)
    {
        $eventId = Session::get('event_id');
        $speakers = EventSpeaker::where('event_id', $eventId);

        if ($request->filter) {
            $speakers->where(function ($q) use ($request) {
                $requestContact = preg_replace("/[^0-9]/", "", $request->filter);
                $q->where('name', 'like', "%{$request->filter}%")
                    ->orWhereLike('contact', $requestContact);
            });
        }
        $speakers = $speakers->paginate(25);
        return view('event.speaker.index', compact('speakers', 'request'));
    }

    public function create(Request $request)
    {
        $eventSpeakerTypes = EventSpeakerType::where('event_id', Session::get('event_id'))->get();
        return view('event.speaker.add', compact('eventSpeakerTypes'));
    }

    public function store(Request $request, Event $event)
    {
        $request->validate([
            'title' => ['required'],
            'name' => ['required', 'string'],
            'country_code' => ['required'],
            'contact' => ['required', 'digits:10'],
            'designation' => ['nullable'],
            'personal_info' => ['nullable'],
            'institute' => ['nullable'],
            'text_1' => ['nullable'],
            'text_2' => ['nullable'],
            'bio' => ['nullable'],
            'event_speaker_type_id' => ['required', 'exists:event_speaker_types,id'],
            'image' => ['nullable', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);

        if (isset($request->image)) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $data['country_code'] = "+" . str_replace('+', '', $data['country_code']);
        $event->eventSpeakers()->create($data);
        return redirect()->route('event.speakers.index')->with('success', 'Speaker Created');
    }

    public function edit(EventSpeaker $speaker)
    {
        $eventSpeakerTypes = EventSpeakerType::where('event_id', Session::get('event_id'))->get();
        return view('event.speaker.edit', compact('speaker', 'eventSpeakerTypes'));
    }

    public function update(EventSpeaker $speaker, Request $request)
    {
        $request->validate([
            'title' => ['required'],
            'name' => ['required', 'string'],
            'country_code' => ['required'],
            'contact' => ['required', 'integer'],
            'designation' => ['nullable'],
            'personal_info' => ['nullable'],
            'institute' => ['nullable'],
            'text_1' => ['nullable'],
            'text_2' => ['nullable'],
            'bio' => ['nullable'],
            'event_speaker_type_id' => ['required', 'exists:event_speaker_types,id'],
            'image' => ['nullable', 'image', 'max:3072'],
        ]);

        $data = $request->except(['_token', '_method']);
        if (isset($request->image)) {
            $data['media_id'] = $this->upload($request->image)->id;
        }
        $data['country_code'] = "+" . str_replace('+', '', $data['country_code']);
        $speaker->update($data);
        return redirect()->route('event.speakers.index')->with('success', 'Speaker Updated');
    }

    public function statusChange(EventSpeaker $speaker)
    {
        $newStatus = ($speaker->status == StatusEnum::ACTIVE) ? StatusEnum::INACTIVE : StatusEnum::ACTIVE;
        $speaker->status = $newStatus;
        $speaker->save();
        return response()->json(["status" => 200, "message" => "Speaker Status Changed"]);
    }

    public function updateIndex(Request $request, EventSpeaker $speaker)
    {
        $request->validate([
            'index' => 'required',
        ]);
        try {
            $speaker->index = $request->index;
            $speaker->update();

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => 'success',
                    'message' => 'Index change successfully',
                ]);
            }
            return redirect()->route('event.speakers.index');
        } catch (\Exception $e) {
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }

    public function destroy(EventSpeaker $speaker)
    {
        $speaker->delete();
        return response()->json(["status" => 200, "message" => "Speaker Deleted"]);
    }

    public function import(Request $request, $eventId)
    {
        try {
            $request->validate([
                'file' => ['required', 'file'],

            ]);
            Excel::import(new EventSpeakerImport($eventId), $request->file('file')->store('app'));

            return redirect()->back()->with('success', 'File imported successfully!');
        } catch (\Exception $e) {
            $this->logError($e, "Speaker Import time Error");
            return redirect()->back()->with(['error', $e->getMessage()]);
        }
    }

}
