<?php

namespace App\Http\Controllers\Frontend;

use App\Enums\EventFormCategoryTypeEnum;
use App\Enums\EventFormFieldEnum;
use App\Enums\EventFormFieldTypeEnum;
use App\Enums\EventRegistrationTypeEnum;
use App\Enums\MailTemplateCampaignEnum;
use App\Enums\MessageCampaignEnum;
use App\Enums\RoleEnum;
use App\Enums\StatusEnum;
use App\Helpers\SendEmailCampaign;
use App\Helpers\Wati;
use App\Helpers\WatiMessage;
use App\Http\Controllers\Controller;
use App\Http\Traits\MediaUpload;
use App\Models\AssociationUser;
use App\Models\City;
use App\Models\Country;
use App\Models\Event;
use App\Models\EventBankDetail;
use App\Models\EventFormCategory;
use App\Models\EventFormSlab;
use App\Models\EventFormSlabCategory;
use App\Models\EventFormWorkshop;
use App\Models\EventMailTemplate;
use App\Models\EventRegistration;
use App\Models\MailTemplateCampaign;
use App\Models\PaymentOrder;
use App\Models\RequestLog;
use App\Models\State;
use App\Models\User;
use App\Rules\DisposableEmailRule;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class EventRegisterController extends Controller
{
    use MediaUpload;

    public function registrationForm(Event $event)
    {
        $currentDate = Carbon::now()->toDateString();
        $eventFormInputFields = $event->eventFormInputFields()
            ->where('visible', EventFormFieldEnum::VISIBLE)
            // ->where('field_type', EventFormFieldTypeEnum::STATIC)
            ->get();
        // Get current active slab
        $activeSlab = EventFormSlab::where('event_id', $event->id)
            ->whereDate('start_date', '<=', $currentDate)
            ->whereDate('end_date', '>=', $currentDate)
            ->first();

        $eventCategories = [];
        // category
        if ($activeSlab) {
            $slabCategoryIds = EventFormSlabCategory::where('event_id', $event->id)
                ->where('event_form_slab_id', $activeSlab->id)
                ->pluck('event_form_category_id')->toArray();
            $eventCategories = EventFormCategory::whereIn('id', $slabCategoryIds)
                ->where('type', EventFormCategoryTypeEnum::CATEGORY)
                ->get();
        }

        $eventWorkshops = $event->eventFormWorkshops()->where('status', StatusEnum::ACTIVE)
            ->whereColumn('no_of_registration', '<', 'registration_limit')
            ->get();

        return view('frontend.event.registration.index', [
            'event' => $event,
            'eventFormInputFields' => $eventFormInputFields,
            'eventCategories' => $eventCategories,
            'eventWorkshops' => $eventWorkshops,
            'eventUpiDetails' => $event->eventUpiDetails
        ]);
    }

    public function register(Event $event, Request $request)
    {
        $data = $request->all();
        RequestLog::create([
            'name' => 'Event Registration Via Form',
            'data' => $data
        ]);

        $extra = $request->except('full_name', '_token', 'email', 'mobile');

        if (isset($data['upload_payment_receipt'])) {
            $mediaId = $this->upload($data['upload_payment_receipt'])->id;
            $extra['upload_payment_receipt'] = $mediaId;
        }

        if (isset($data['category'])) {
            $currentDate = Carbon::now()->toDateString();
            $activeSlab = EventFormSlab::where('event_id', $event->id)
                ->whereDate('start_date', '<=', $currentDate)
                ->whereDate('end_date', '>=', $currentDate)
                ->first();

            // category data
            if ($activeSlab) {
                $slabCategory = EventFormSlabCategory::where('event_id', $event->id)
                    ->where('event_form_slab_id', $activeSlab->id)
                    ->where('event_form_category_id', $data['category'])
                    ->first();

                if ($slabCategory) {
                    $category = EventFormCategory::find($data['category']);
                    $extra['category_data']['category_id'] = $data['category'];
                    $extra['category_data']['category_name'] = $category->name ?? null;
                    $extra['category_data']['category_amount'] = $slabCategory->amount;
                }
            }
        }

        if (isset($data['workshop'])) {
            $workshop = EventFormWorkshop::find($data['workshop']);
            if ($workshop) {
                $extra['workshop_data']['workshop_id'] = $data['workshop'];
                $extra['workshop_data']['workshop_name'] = $workshop->name ?? null;
                $extra['workshop_data']['workshop_amount'] = $workshop->registration_fee;
            }
        }

        if (isset($data['register_accompany']) && $data['register_accompany'] == 'Yes') {
            $activeSlab = EventFormSlab::where('event_id', $event->id)
                ->whereDate('start_date', '<=', $currentDate)
                ->whereDate('end_date', '>=', $currentDate)
                ->first();

            // accompany Person
            $accompanyCategory = EventFormCategory::where('event_id', $event->id)->where('type', EventFormCategoryTypeEnum::OTHER)->first();
            if ($accompanyCategory) {
                $accompanySlab = EventFormSlabCategory::where('event_form_slab_id', $activeSlab->id)
                    ->where('event_form_category_id', $accompanyCategory->id)
                    ->first();

                if ($accompanySlab) {
                    $extra['accompany_data']['accompany_person'] = $data['accompany_person'] ?? null;
                    $extra['accompany_data']['accompany_name'] = $data['accompany_name'] ?? null;
                    $extra['accompany_data']['accompany_amount'] = $accompanySlab->amount;
                    $extra['accompany_data']['accompany_total_amount'] = $accompanySlab->amount * (int) $data['accompany_person'];
                }
            }
        }

        if ($data['email'] && $user = User::where('email', $data['email'])->first()) {
            if ($user->eventRegistrations()->where('event_id', $event->id)->exists()) {
                return back()
                    ->withErrors([
                        'error' => "Email already registered",
                    ])
                    ->withInput();
            }
            $user->name = $data['full_name'];
            $user->contact = $data['mobile'];
            $user->extra = $extra;
            $user->save();
            Log::info('USER UPDATE REGISTRATION email : ' . json_encode($user));
            return $this->handleRegisteredUser($user, $extra, $event);
        }

        $password = substr($data['mobile'], -4);

        $country = Country::where("name", $data['country'] ?? null)->first();

        $user = User::create([
            'name' => $data['full_name'],
            'email' => $data['email'],
            'contact' => $data['mobile'],
            'password' => Hash::make($password),
            'pin' => $password,
            'country_code' => $country ? '+' . $country->country_code : '+91',
            'status' => StatusEnum::ACTIVE,
            'role' => RoleEnum::MEMBER,
            'association_id' => $event->association_id ?? null,
            'extra' => $extra
        ]);

        AssociationUser::UpdateOrCreate([
            'user_id' => $user->id,
            'association_id' => $event->association_id,
        ]);

        $userDetail['member_id'] = $user->id;
        $userDetail["association_id"] = $event->association_id ?? null;
        $user->memberDetail()->create($userDetail);

        Log::warning("Event Register: {$user->contact} - $password");
        return $this->handleRegisteredUser($user, $extra, $event);
    }

    protected function handleRegisteredUser(User $user, $data, Event $event)
    {
        $eventFee = $data['total_amount'];

        if ($eventFee != 0 && $data['payment_mode'] != EventRegistrationTypeEnum::OFFLINE_REGISTRATION) {
            return $this->createOrder($event, $user, $eventFee);
        }

        if ($eventFee != 0 && $data['payment_mode'] == EventRegistrationTypeEnum::OFFLINE_REGISTRATION) {
            if (env('APP_ENV') == 'production') {
                $this->sendMessageBeforeConfirmation($event,$user);
            }
        }

        $referralId = Str::uuid();

        $order = PaymentOrder::create([
            'event_id' => $event->id,
            'user_id' => $user->id,
            'order_id' => $data['transaction_id'] ?? null,
            'amount' => $eventFee,
            'currency' => 'INR',
            'referral_code' => $referralId,
            'payment_mode' => 'offline',
            'member_data' => $data,
        ]);
        Log::info('ORDER CREATION OFFLINE :' . json_encode($order));

        //skip pay page
        Log::info("callbackkk:" . $event->eventPaymentMode->getCallback($referralId));
        return response()->json(['payment_link' => $event->eventPaymentMode->getCallback($referralId)]);

        //without skipping pay page
        // $eventRegistration = EventRegistration::create([
        //     'event_id' => $event->id,
        //     'user_id' => $user->id,
        //     'association_id' => $event->association_id,
        //     'extra' => $data,
        //     'paid_amount' => $eventFee
        // ]);

        // return view('frontend.event.registration.thanks', [
        //     'register' => true,
        //     'event' => $event,
        //     'contact' => $user->contact,
        //     'amount' => $eventFee,
        //     'user' => $user,
        // ]);
    }

    private function createOrder(Event $event, User $user, $eventRegistrationFee)
    {
        $appKey = $event->eventPaymentMode->app_key;
        $client = $event->eventPaymentMode->getPaymentClient();

        try {
            $referralId = Str::uuid();

            $callbackURL = $event->eventPaymentMode->getCallback($referralId);

            $orderData = [
                "callback_url" => $callbackURL,
                "referral_id" => $referralId
            ];
            $order = $client->createOrder($event->code, $eventRegistrationFee, $user, $orderData);
            $paymentOrder = PaymentOrder::create([
                'event_id' => $event->id,
                'user_id' => $user->id,
                'order_id' => $order["id"],
                'amount' => $eventRegistrationFee,
                'currency' => 'INR',
                'referral_code' => $referralId,
                "app_key" => $appKey ?? null,
                "member_data" => $user->extra ?? null,
                'payment_mode' => $event->eventPaymentMode->name ?? null,
            ]);
            Log::info('ORDER CREATION ONLINE :' . json_encode($paymentOrder));

            //skip the event pay page
            return response()->json([
                'payment_mode' => $event->eventPaymentMode->name ?? null,
                'order_id' => $order["id"],
                'amount' => $eventRegistrationFee,
                'api_key' => $appKey,
                'callback_url' => $callbackURL,
                'name' => $user->name,
                'email' => $user->email,
                'contact' => $user->contact,
                'payment_link' => $order["longurl"] ?? null,
            ]);

            // return view('frontend.event.registration.event-pay', [
            //     'event' => $event,
            //     'order_id' => $order["id"],
            //     'order_refer_id' => $paymentOrder->referral_code,
            //     'name' => $user->name,
            //     'email' => $user->email,
            //     'contact' => $user->contact,
            //     'razorpay_api_key' => $appKey,
            //     'payment_link' => $order["longurl"] ?? null,
            //     'callback_link' => $callbackURL,
            //     'eventRegistrationFee' => $eventRegistrationFee,
            //     "memberData" => $memberData
            // ]);  
        } catch (Exception $e) {
            Log::error("Event Registration Exception: " . $e->getMessage());
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function thanks(Request $request, $orderRef = null)
    {
        Log::info("Thanks Order: {$orderRef} Method: " . $request->getMethod());

        RequestLog::create([
            'name' => 'Event Payment Verify',
            'data' => array_merge($request->all(), ["order_ref_id" => $orderRef]),
        ]);

        $paymentOrder = PaymentOrder::getForReferralCode($orderRef);

        Log::info("Grid Order Found: " . ($paymentOrder->id ?? "NULL"));

        if (!$paymentOrder) {
            Log::info("Event Thanks: Refer code missing");
            return redirect()->back()->with('error', "Payment Information not found");
        }

        $event = $paymentOrder->event;
        $user = $paymentOrder->user;

        $data = $request->all();
        Log::info("Thanks Order Data: " . json_encode($data));

        $eventForm = null;

        if (!empty($paymentOrder->payment_id)) {
            Log::info("Event Thanks: Refer code duplicate");
            $eventRegistration = $user->eventRegistrations()
                ->where('event_id', $paymentOrder->event_id)
                ->first();
            return view('frontend.event.registration.online-thanks', [
                'type' => EventRegistrationTypeEnum::ONLINE,
                'register' => true,
                'event' => $event,
                'pin' => $user->pin_last_updated_at == null ? substr($user->contact, -4) : null,
                'contact' => $user->contact,
                'amount' => $paymentOrder->amount,
                'user' => $user,
            ]);
        }

        if ($paymentOrder->payment_mode == EventRegistrationTypeEnum::OFFLINE) {
            Log::info("Event Thanks: Offline Registration");
            return view('frontend.event.registration.offline-thanks', [
                'type' => EventRegistrationTypeEnum::OFFLINE,
                'register' => true,
                'event' => $event,
                'pin' => $user->pin_last_updated_at == null ? substr($user->contact, -4) : null,
                'contact' => $user->contact,
                'amount' => $paymentOrder->amount,
                'user' => $user,
            ]);
        }

        Log::info("Verifying the Order: {$event->payment_mode}");
        try {
            $apiClient = $event->eventPaymentMode->getPaymentClient();
            $response = $apiClient->getPaymentDetails($data);

            Log::info("Verifying the Order: {$event->eventPaymentMode->name} " . json_encode($response));

            if ($response["status"] !== 'paid') {
                Log::info("Event Thanks: Refer code missing");
                return redirect()->back()->with('error', "Payment Information not found");
            }

            $paymentOrder->payment_id = $response["payment_id"];
            $paymentOrder->signature = $response["signature"];
            $paymentOrder->status = "SUCCESS";
            $paymentOrder->save();
            $paymentAmount = $response['amount'];

            if (isset($paymentOrder->member_data['workshop'])) {
                $workshop = EventFormWorkshop::where('id', $paymentOrder->member_data['workshop'])
                    ->where('event_id', $event->id)
                    ->first();
                if ($workshop) {
                    $workshop->no_of_registration += 1;
                    $workshop->update();
                }
            }

            $eventRegistration = EventRegistration::create([
                'event_id' => $event->id,
                'user_id' => $user->id,
                'association_id' => $event->association_id,
                'payment_info' => $request->all(),
                'extra' => $user->extra,
                'paid_amount' => $paymentAmount,
            ]);

            if (env('APP_ENV') == 'production') {
                $this->sendMessage($eventRegistration);
            }

            return view('frontend.event.registration.online-thanks', [
                'type' => EventRegistrationTypeEnum::ONLINE,
                'register' => true,
                'event' => $event,
                'amount' => $paymentAmount,
                'pin' => substr($user->contact, -4),
                'user' => $user,
                'eventForm' => $eventForm,
                'contact' => $user->contact
            ]);
        } catch (Exception $e) {
            Log::info("Event Thanks: Payment Detail Fetch Failed - " . $e->getMessage());
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function getTotalAmount(Request $request, Event $event)
    {
        $categoryId = $request->input('category_id');
        $workshopId = $request->input('workshop_id');
        $accompanyCount = (int) $request->input('accompany_person', 0);
        $currentDate = Carbon::now()->toDateString();
        $amount = 0;

        // Get current active slab
        $activeSlab = EventFormSlab::where('event_id', $event->id)->whereDate('start_date', '<=', $currentDate)
            ->whereDate('end_date', '>=', $currentDate)
            ->first();

        // category
        if ($activeSlab) {
            $slabCategory = EventFormSlabCategory::where('event_id', $event->id)->where('event_form_slab_id', $activeSlab->id)
                ->where('event_form_category_id', $categoryId)
                ->first();

            if ($slabCategory) {
                $amount += $slabCategory->amount;
            }
        }

        // accompany Person
        $accompanyCategory = EventFormCategory::where('event_id', $event->id)->where('type', EventFormCategoryTypeEnum::OTHER)->first();

        if ($accompanyCategory) {
            $accompanySlab = EventFormSlabCategory::where('event_form_slab_id', $activeSlab->id)
                ->where('event_form_category_id', $accompanyCategory->id)
                ->first();

            if ($accompanySlab && $accompanyCount > 0) {
                $amount += $accompanyCount * $accompanySlab->amount;
            }
        }

        // workshop
        if ($workshopId) {
            $workshop = EventFormWorkshop::where('event_id', $event->id)->where('id', $workshopId)->first();
            if ($workshop && $workshop->registration_fee) {
                $amount += $workshop->registration_fee;
            }
        }

        return response()->json([
            'status' => true,
            'amount' => $amount,
        ]);
    }

    public function getStates($countryId)
    {
        $states = State::where('country_id', $countryId)->get();
        return response()->json($states);
    }

    public function getCities($stateId)
    {
        $cities = City::where('state_id', $stateId)->get();
        return response()->json($cities);
    }

    protected function sendMessageBeforeConfirmation(Event $event,User $user)
    {

        if (!$user || !$event) {
            return;
        }

        // send mail
        if ($event->eventConfig && $event->eventConfig->offlineMailTemplateBeforeConfirmationActive) {
            Log::info('SEND EMAIL START BEFORE CONFIRMATION');
            $disposableEmailRule = new DisposableEmailRule();
            if (!$disposableEmailRule->isDisposable($user->email)) {
                $this->createMailCampaign($user, $event->eventConfig->offlineMailTemplateBeforeConfirmationActive);
            } else {
                Log::info('USER MAIL IS DISPOSABLE: ' . $user->email);
            }
            Log::info('SEND EMAIL END');
        }

        if ($event->eventConfig && $event->eventConfig->offline_wati_template_before_confirmation) {
            WatiMessage::sendMessageIfRequired($user, $event, $event->eventConfig->offline_wati_template_before_confirmation);
        }

        return;
    }
    
    protected function sendMessage(EventRegistration $eventRegistration)
    {
        $event = $eventRegistration->event;
        $user = $eventRegistration->user;
        if (!$user || !$event) {
            return;
        }

        // send mail
        if ($event->eventConfig && $event->eventConfig->onlineMailTemplateActive) {
            Log::info('SEND EMAIL START');
            $disposableEmailRule = new DisposableEmailRule();
            if (!$disposableEmailRule->isDisposable($user->email)) {
                $this->createMailCampaign($user, $event->eventConfig->onlineMailTemplateActive);
            } else {
                Log::info('USER MAIL IS DISPOSABLE: ' . $user->email);
            }
            Log::info('SEND EMAIL END');
        }

        if ($event->eventConfig && $event->eventConfig->online_wati_template) {
            WatiMessage::sendMessageIfRequired($user, $event, $event->eventConfig->online_wati_template);
        }

        return;
    }

    protected function createMailCampaign(User $user, ?EventMailTemplate $mailTemplate = null)
    {
        if ($mailTemplate === null) {
            Log::info('Active EventMailTemplate Not Found.');
            return;
        }
        $usersArray[] = $user->id;
        $mailTemplateCampaign = new MailTemplateCampaign();
        $mailTemplateCampaign->template_name = $mailTemplate->id;
        $mailTemplateCampaign->scheduled_date = date('Y-m-d');
        $mailTemplateCampaign->scheduled_time = date('H:i');
        $mailTemplateCampaign->total_users = count($usersArray);
        $mailTemplateCampaign->users = $usersArray;
        $mailTemplateCampaign->success_users = [];
        $mailTemplateCampaign->fail_users = [];
        $mailTemplateCampaign->status = MailTemplateCampaignEnum::PENDING;
        $mailTemplateCampaign->schedule_type = 'send_now';
        $mailTemplateCampaign->save();
        $host = request()->getHost();
        (new SendEmailCampaign())->sendEventEmailCampaign($host, $mailTemplateCampaign);
        return;
    }

    public function checkMailExists(Request $request, Event $event)
    {
        if ($user = User::where('email', $request->email)->first()) {
            if ($user->eventRegistrations()->where('event_id', $event->id)->first()) {
                return response()->json(['exists' => true]);
            }
        }
        return response()->json(['exists' => false]);
    }

    public function checkTransactionExists(Request $request)
    {
        if (PaymentOrder::where('order_id', $request->transactionId)->first()) {
            return response()->json(['exists' => true]);
        }
        return response()->json(['exists' => false]);
    }
}
