<?php

namespace App\Jobs;

use App\Models\Event;
use App\Models\UserFcmToken;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;

class SendEventNotificationJob implements ShouldQueue
{
    use Queueable;

    protected $title;
    protected $body;
    protected $eventId;

    /**
     * Create a new job instance.
     */
    public function __construct($title, $body, $eventId)
    {
        $this->title = $title;
        $this->body = $body;
        $this->eventId = $eventId;
    }

    /**
     * Execute the job.
     */
    public function handle()
    {
        try {
            $tokens = UserFcmToken::where('event_id', $this->eventId)->pluck('token')->toArray();

            Log::info("Sending notifications for event ID: {$this->eventId}");

            if (!file_exists(storage_path('credentials/firebase-credentials.json'))) {
                Log::error("Firebase credentials file not found");
                return false;
            }

            $eventCode = [
                2908,
                1012,
                6734,
                5432,
            ];

            $eventIdsForSecondFirebaseCredentail = Event::whereIn('code', $eventCode)->pluck('id')->toArray();
            if (in_array($this->eventId, $eventIdsForSecondFirebaseCredentail)) {
                $factory = (new Factory())->withServiceAccount(storage_path('credentials/event-second-firebase-credentials.json'));
            } else {
                $factory = (new Factory())->withServiceAccount(storage_path('credentials/firebase-credentials.json'));
            }

            $messaging = $factory->createMessaging();

            $notification = Notification::create($this->title, $this->body);

            $chunks = array_chunk($tokens, 500);

            foreach ($chunks as $chunk) {
                $message = CloudMessage::new()->withNotification($notification);
                try {
                    // Send each batch of tokens
                    $messaging->sendMulticast($message, $chunk);
                    Log::info("Notifications sent successfully for event ID: {$this->eventId}");
                } catch (\Exception $e) {
                    Log::error("Send Notification Error for event ID {$this->eventId}: " . $e->getMessage());
                }
            }

            return true;
        } catch (\Exception $e) {
            Log::error("Send Notification Error for event ID {$this->eventId}: " . $e->getMessage());
            return false;
        }
    }
}
