<?php

namespace App\Models;

use App\Enums\AdminLevel;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Event extends Model
{
    use HasFactory;

    const OTHERS = -1;
    const CONFERENCE = 1;
    const WEBINAR = 2;
    protected $fillable = [
        'association_id',
        'title',
        'type',
        'description',
        'date',
        'status',
        'link',
        'banner_id',
        'message_template_id',
        'mail_template_id',
        'code',
        'wati_template',
        'badge_media_id',
    ];

    protected $casts = [
        'date' => 'datetime:Y-m-d',
    ];

    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    public function scopeFilterEvent($query)
    {
        if (request()->filled('title')) {
            $query->where('title', 'like', '%' . request('title') . '%');
        }
        if (request()->filled('type')) {
            $query->where('type', request('type'));
        }
        if (request()->filled('status')) {
            $query->where('status', request('status'));
        }
        if (request()->filled('date')) {
            $query->whereDate('date', date('Y-m-d', strtotime(request('date'))));
        }

        if (request()->filled('filter')) {
            $filter = request('filter');
            if ($filter === 'past') {
                $query->whereDate('date', '<', now());
            } elseif ($filter === 'upcoming') {
                $query->whereDate('date', '>=', now());
            }
        }
    }

    /**
     * @return BelongsTo
     */
    public function banner(): BelongsTo
    {
        return $this->belongsTo(Media::class, 'banner_id');
    }

    /**
     * @return BelongsTo
     */
    public function badge(): BelongsTo
    {
        return $this->belongsTo(Media::class, 'badge_media_id');
    }

    /**
     * @return BelongsTo
     */
    public function mailTemplate()
    {
        return $this->belongsTo(MailTemplate::class, 'mail_template_id');
    }

    /**
     * @return BelongsTo
     */
    public function messageTemplate()
    {
        return $this->belongsTo(MessageTemplate::class, 'message_template_id');
    }

    /**
     * @return HasMany
     */
    public function eventRegistrations()
    {
        return $this->hasMany(EventRegistration::class);
    }

    public function eventManager()
    {
        return $this->hasOne(EventManager::class, 'event_id');
    }

    public function eventManagers()
    {
        return $this->hasMany(EventManager::class, 'event_id')->whereHas('admin', function ($query) {
            $query->where('role', AdminLevel::EVENT_COORDINATOR);
        });
    }

    /**
     * @return HasMany
     */
    public function eventDays(): HasMany
    {
        return $this->hasMany(EventDay::class, 'event_id');
    }

    public function eventEPosterCategories(): HasMany
    {
        return $this->hasMany(EventEPosterCategory::class, 'event_id');
    }

    public function eventSpeakerTypes(): HasMany
    {
        return $this->hasMany(EventSpeakerType::class, 'event_id');
    }

    public function eventExhibitorCategories(): HasMany
    {
        return $this->hasMany(EventExhibitorCategory::class, 'event_id');
    }

    public function eventContactUsTypes(): HasMany
    {
        return $this->hasMany(EventContactUsType::class, 'event_id');
    }

    public function eventImportantDates(): HasMany
    {
        return $this->hasMany(EventImportantDate::class, 'event_id');
    }

    public function eventImportantLinks(): HasMany
    {
        return $this->hasMany(EventImportantLink::class, 'event_id');
    }

    public function eventDocumentLinks(): HasMany
    {
        return $this->hasMany(EventDocumentList::class, 'event_id');
    }

    public function eventEPosters(): HasMany
    {
        return $this->hasMany(EventEPoster::class, 'event_id');
    }

    public function bannerImages()
    {
        return $this->morphMany(ScrollerImage::class, 'imageable')->where('type', ScrollerImage::TYPE_BANNER);
    }

    public function noticeImages()
    {
        return $this->morphMany(ScrollerImage::class, 'imageable')->where('type', ScrollerImage::TYPE_NOTICE);
    }

    /**
     * @return HasMany
     */
    public function eventHalls(): HasMany
    {
        return $this->hasMany(EventHall::class, 'event_id');
    }

    /**
     * @return HasMany
     */
    public function eventFaculties(): HasMany
    {
        return $this->hasMany(EventFaculty::class, 'event_id');
    }

    /**
     * @return HasMany
     */
    public function eventSpeakers(): HasMany
    {
        return $this->hasMany(EventSpeaker::class, 'event_id');
    }

    /**
     * @return HasMany
     */
    public function eventSessions(): HasMany
    {
        return $this->hasMany(EventSession::class, 'event_id');
    }

    /**
     * @return HasMany
     */
    public function eventDesignations(): HasMany
    {
        return $this->hasMany(EventDesignation::class, 'event_id');
    }

    public function eventSessionUsers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'event_session_users')
            ->withTimestamps();
    }

    public function eventContactUs(): HasMany
    {
        return $this->hasMany(EventContactUs::class, 'event_id');
    }

    public function eventNotifications(): HasMany
    {
        return $this->hasMany(Notification::class, 'event_id');
    }

    public function eventAppHomeButtons(): HasMany
    {
        return $this->hasMany(EventAppHomeButton::class, 'event_id');
    }

    public function eventAppSponsorLinks(): HasMany
    {
        return $this->hasMany(EventAppSponsorLink::class, 'event_id');
    }

    public function eventDashboardSponsorLinks(): HasMany
    {
        return $this->hasMany(EventDashboardSponsorLink::class, 'event_id');
    }

    public function eventFormCategories(): HasMany
    {
        return $this->hasMany(EventFormCategory::class, 'event_id');
    }

    public function eventFormSlabs(): HasMany
    {
        return $this->hasMany(EventFormSlab::class, 'event_id');
    }

    public function eventFormWorkshops(): HasMany
    {
        return $this->hasMany(EventFormWorkshop::class, 'event_id');
    }

    public function eventFormInputFields(): HasMany
    {
        return $this->hasMany(EventFormInputField::class, 'event_id');
    }

    public function eventBankDetails()
    {
        return $this->hasOne(EventBankDetail::class, 'event_id');
    }

    public function eventUpiDetails()
    {
        return $this->hasOne(EventUpiDetail::class, 'event_id');
    }

    public function eventPaymentMode()
    {
        return $this->hasOne(EventPaymentMode::class, 'event_id');
    }

    public function eventMailTemplates()
    {
        return $this->hasMany(EventMailTemplate::class, 'event_id');
    }

    public function eventConfig()
    {
        return $this->hasOne(EventConfig::class, 'event_id');
    }

    public static function getFileInput($id)
    {
        $receipt = Media::find($id);
        if ($receipt && $receipt->path) {
            return $receipt->path;
        }
        return null;
    }
}
