<?php

namespace App\Models;

use App\Enums\MailTemplateCampaignEnum;
use App\Enums\StatusEnum;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class MailTemplateCampaign extends Model
{
    use HasFactory;

    protected $fillable = [
        'template_name',
        'scheduled_date',
        'scheduled_time',
        'scheduled_at',
        'total_users',
        'users',
        'success_users',
        'fail_users',
        'status',
        'schedule_type',
    ];
    protected $appends = [
        'email_template_name',
        'email_template_subject',
        'event_email_template_name',
        'event_email_template_subject',
    ];

    protected $casts = [
        'users' => 'array',
        'success_users' => 'array',
        'fail_users' => 'array',
    ];

    public function emailTemplate()
    {
        return $this->belongsTo(MailTemplate::class, 'template_name', 'template_id');
    }

    public function eventEmailTemplate()
    {
        return $this->belongsTo(EventMailTemplate::class, 'template_name');
    }

    function getEmailTemplateNameAttribute()
    {
        $data = $this->emailTemplate;
        return $data->name ?? '';
    }

    function getEmailTemplateSubjectAttribute()
    {
        $data = $this->emailTemplate;
        return $data->subject ?? '';
    }
    function getEventEmailTemplateNameAttribute()
    {
        $data = $this->eventEmailTemplate;
        return $data->name ?? '';
    }

    function getEventEmailTemplateSubjectAttribute()
    {
        $data = $this->eventEmailTemplate;
        return $data->subject ?? '';
    }

    public function handleMailSuccess($userId)
    {
        $users = $this->success_users;
        $users[] = $userId;
        $this->success_users = $users;

        $this->setCampaignStatus();
        $this->save();
    }

    public function handleMailFailed($userId)
    {
        $users = $this->fail_users;
        $users[] = $userId;
        $this->fail_users = $users;

        $this->setCampaignStatus();
        $this->save();
    }

    public function setCampaignStatus()
    {
        if ($this->total_users == count($this->success_users) + count($this->fail_users)) {
            $this->status = MailTemplateCampaignEnum::SUCCESS;
        }
    }

    public function getSuccessPercentageAttribute()
    {
        $howManySuccess = is_array($this->success_users) ? count($this->success_users) : 0;
        if ($this->total_users == 0) {
            return 0;
        }
        return ($howManySuccess * 100) / $this->total_users;
    }

    public function getFailPercentageAttribute()
    {
        $howManyFail = is_array($this->fail_users) ? count($this->fail_users) : 0;
        if ($this->total_users == 0) {
            return 0;
        }
        return ($howManyFail * 100) / $this->total_users;
    }

    public function getSentUsersAttribute()
    {
        if (!empty($this->success_users)) {
            return User::whereIn('id', $this->success_users)->get();
        }
        return null;
    }

    public function getFailedUsersAttribute()
    {
        if (!empty($this->fail_users)) {
            return User::whereIn('id', $this->fail_users)->get();
        }
        return null;
    }
}
