<?php

namespace App\Models;

use App\Enums\MessageCampaignEnum;
use App\Jobs\Meta\StartMessageCampaign;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class MessageCampaign extends Model
{
    use HasFactory;

    const PENDING = 1;
    const SUCCESS = 2;

    protected $fillable = [
        "template_name",
        "campaign_unique_id",
        "scheduled_date",
        "scheduled_time",
        "scheduled_at",
        "total_users",
        "users",
        "success_users",
        "fail_users",
        "status",
        "schedule_type",
        "sent_users",
        "delivered_users",
        "read_users",
        "message_template_id"
    ];

    protected $casts = [
        'users' => 'array',
        'success_users' => 'array',
        'fail_users' => 'array',
        'sent_users' => 'array',
        'delivered_users' => 'array',
        'read_users' => 'array',
    ];

    public function messageTemplate(): BelongsTo
    {
        return $this->belongsTo(MessageTemplate::class, 'message_template_id');
    }

    public function start($host)
    {
        StartMessageCampaign::dispatchSync($host, $this->id);
    }

    public function handleSuccess($userId)
    {
        $users = $this->success_users;
        $users[] = $userId;
        $this->success_users = $users;

        $this->setCampaignStatus();
        $this->save();
    }

    public function handleStatus($userId, $type)
    {
        $status = $type . "_users";
        $users = $this->$status;
        $users[] = $userId;

        $this->$status = $users;
        $this->save();
    }

    public function handleFailed($userId)
    {
        $users = $this->fail_users;
        $users[] = $userId;
        $this->fail_users = $users;

        $this->setCampaignStatus();
        $this->save();
    }

    public function setCampaignStatus()
    {
        $failUsers = is_array($this->fail_users) ? $this->fail_users : [];
        $successUsers = is_array($this->success_users) ? $this->success_users : [];
        if ($this->total_users == count($failUsers) + count($successUsers)) {
            $this->status = MessageCampaignEnum::SUCCESS;
        }
    }

    public function campaignUsers()
    {
        return $this->hasMany(MessageCampaignUser::class, "message_campaign_id");
    }

    public function getReadPercentageAttribute()
    {
        $howManyread = is_array($this->read_users) ? count($this->read_users) : 0;
        if ($this->total_users == 0) {
            return 0;
        }
        return ($howManyread * 100) / $this->total_users;
    }

    public function getSentPercentageAttribute()
    {
        $howManysent = is_array($this->sent_users) ? count($this->sent_users) : 0;
        if ($this->total_users == 0) {
            return 0;
        }
        return ($howManysent * 100) / $this->total_users;
    }

    public function getDeliveredPercentageAttribute()
    {
        $howManydelivered = is_array($this->delivered_users) ? count($this->delivered_users) : 0;
        if ($this->total_users == 0) {
            return 0;
        }
        return ($howManydelivered * 100) / $this->total_users;
    }

    public function getSentUsersAttribute()
    {
        if (!empty($this->success_users)) {
            return User::whereIn('id', $this->success_users)->get();
        }
        return null;
    }

    public function getFailedUsersAttribute()
    {
        if (!empty($this->fail_users)) {
            return User::whereIn('id', $this->fail_users)->get();
        }
        return null;
    }
}
