<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;

use App\Http\Traits\MetaProperties;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    use HasFactory, Notifiable, HasApiTokens;

    use MetaProperties;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'unique_id',
        'country_code',
        'contact',
        'password',
        'status',
        'role',
        'pin',
        'association_code',
        'association_id',
        'extra',
        'wati_id',
        'wati_wa_id'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'pin',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'extra' => 'array',
        ];
    }
    protected $appends = ['full_mobile_number'];

    public function roles()
    {
        return $this->hasMany(MemberRole::class, 'association_id');
    }

    public function committees()
    {
        return $this->hasMany(Committee::class, 'association_id');
    }

    public function memberDetail(): HasOne
    {
        return $this->hasOne(MemberDetail::class, 'member_id');
    }

    public function committeeMember(): HasOne
    {
        return $this->hasOne(CommitteeMember::class, 'member_id');
    }

    public function eventCommitteeMember(): HasOne
    {
        return $this->hasOne(EventCommitteeMember::class, 'member_id');
    }

    public function getFullMobileNumberAttribute()
    {
        return $this->country_code . $this->contact;
    }

    public function postLike(): HasOne
    {
        return $this->hasOne(PostLike::class, 'user_id');
    }

    /**
     * @return HasMany
     */
    public function posts(): HasMany
    {
        return $this->hasMany(Post::class, 'user_id');
    }

    /**
     * @return HasMany
     */
    public function comments(): HasMany
    {
        return $this->hasMany(PostComment::class, 'user_id');
    }

    /**
     * @return HasMany
     */
    public function eventRegistrations(): HasMany
    {
        return $this->hasMany(EventRegistration::class, 'user_id');
    }

    public function associationUsers(): HasMany
    {
        return $this->hasMany(AssociationUser::class, 'user_id')->with('association');
    }

    public function associations(): HasMany
    {
        return $this->hasMany(AssociationUser::class, 'user_id')->with('association');
    }

    /**
     * @return BelongsToMany
     */
    public function bookmarkedPosts(): BelongsToMany
    {
        return $this->belongsToMany(Post::class, 'post_bookmarks', 'user_id', 'post_id')
            ->withTimestamps();
    }

    public function bookmarkedSessions(): BelongsToMany
    {
        return $this->belongsToMany(EventSession::class, 'event_session_bookmarks', 'user_id', 'event_session_id')
            ->withTimestamps();
    }

    // public function events(): BelongsToMany
    // {
    //     return $this->belongsToMany(Event::class, 'event_session_users')
    //         ->withPivot('designation')
    //         ->withTimestamps();
    // }

    //_________________________________ ATTRIBUTES ______________________________________
    //___________________________________________________________________________________

    public function getRoleNameAttribute()
    {
        if ($this->memberDetail->role) {
            return $this->memberDetail->role->name;
        }
        return '-';
    }

    public function getCommitteeNameAttribute()
    {
        if (isset($this->committeeMember)) {
            return isset($this->committeeMember->committee) ? $this->committeeMember->committee->name : '-';
        }
        return '-';
    }

    // ____________________________________________________________________________________________________________
    // _______________________________________ END ATTRIBUTES _____________________________________________________
}
