<?php

namespace App\Services;

use App\Helpers\Wati;
use App\Jobs\SendEventMailJob;
use App\Models\OtpRequest;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class OTPService
{
    private OtpRequest $otpRequest;
    private string $otp;

    public function generateOTP(User $user): static
    {
        $this->otp = rand(100000, 999999);

        $this->otpRequest = OtpRequest::create([
            'user_id' => $user->id,
            'country_code' => $user->country_code ?? '+91',
            'contact' => $user->contact,
            'email' => $user->email,
            'otp' => Hash::make($this->otp),
        ]);

        return $this;
    }

    public function sendEmailOTP(): void
    {
        if (!$this->otpRequest->email) {
            return;
        }

        // Use $this->otp now
        $fromMail = 'Vd7bT@example.com';
        $subject = 'OTP';
        $templateId = 1;
        $dynamicData = ['otp' => $this->otp];

        dispatch(new SendEventMailJob(
            ['Email' => $fromMail],
            [['Email' => $this->otpRequest->email]],
            $subject,
            $templateId,
            $dynamicData,
            []
        ));
    }

    public function sendWatiCode()
    {
        if (!$this->otpRequest->wati_contact) {
            return false;
        }

        $watiToken = config('wati.token');
        $watiEndPointUrl = config('wati.endpoint');
        $wati = new Wati($watiToken, $watiEndPointUrl);

        $response = $wati->createContact("{$this->otpRequest->wati_contact}", "");

        if ($response && isset($response['contact'])) {
            $tempName = "vbnotp";
            $wId = $response['contact']['wAid'];

            $params = [
                [
                    "name" => "1", // otp
                    "value" => $this->otp,
                ],

            ];

            $response = $wati->sendGeneralWatiMessage($wId, time(), $tempName, $params);

            return $response && $response['result'] == true;
        }
        return false;
    }

    public function verifyOTP($contact, string $otp)
    {
        $latest = OtpRequest::where('contact', $contact)
            ->where('status', OtpRequest::STATUS_PENDING)
            ->latest()
            ->first();

        if (!$latest) {
            return ['success' => false, 'message' => 'No OTP request found'];
        }

        $otpLifetimeMinutes = 10;

        if (Carbon::parse($latest->created_at)->addMinutes($otpLifetimeMinutes)->isPast()) {
            $latest->update(['status' => OtpRequest::STATUS_EXPIRED]);
            return ['success' => false, 'message' => 'OTP expired'];
        }

        if ($latest->attempt >= 5) {
            return ['success' => false, 'message' => 'Too many failed attempts. Please wait before retrying.'];
        }

        if (!Hash::check($otp, $latest->otp)) {
            $latest->increment('attempt');
            return ['success' => false, 'message' => 'Invalid OTP'];
        }

        $latest->update([
            'status' => OtpRequest::STATUS_VERIFIED,
            'verified_at' => now(),
        ]);

        return ['success' => true, 'message' => 'OTP verified successfully'];
    }

}